#include "ru_yandex_solomon_MetricArchiveNative.h"
#include "java_objects.h"

#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>

using namespace NSolomon;
using namespace NStockpile;

namespace {

TMetricArchive Decode(EFormat format, NJava::TByteArray array) {
    TCodecInput in{array.Data(), static_cast<size_t>(array.Size())};
    TMetricArchiveCodec codec{format};
    return codec.Decode(&in);
}

TBuffer Encode(const TMetricArchive& archive) {
    TCodecOutput out{16};
    TMetricArchiveCodec codec{archive.Format()};
    codec.Encode(archive, &out);
    return out.TakeBuffer();
}

} // namespace

jobject Java_ru_yandex_solomon_MetricArchiveNative_decode(JNIEnv* jenv, jclass, jobject javaFormat, jbyteArray buffer) {
    try {
        NJava::TStockpileFormatClass formatClass{jenv};
        EFormat format = static_cast<EFormat>(formatClass.Convert(javaFormat));

        auto archive = Decode(format, NJava::TByteArray{jenv, buffer});
        const auto& header = archive.Header();

        NJava::TMetricTypeClass metricTypeClass{jenv};
        jobject type = metricTypeClass.ForNumber(static_cast<jint>(header.Type));

        NJava::TMetricHeaderClass metricHeaderClass{jenv};
        jobject javaHeader = metricHeaderClass.New(
                static_cast<jlong>(header.DeleteBefore.MilliSeconds()),
                static_cast<jint>(header.Owner.ProjectId),
                static_cast<jint>(header.Owner.ShardId),
                static_cast<jint>(header.DecimPolicyId),
                type);

        NJava::THeapBitBufClass heapBitBufClass{jenv};
        jobject bitBuf = heapBitBufClass.New(
                NJava::ToByteArray(jenv, archive.Data()),
                static_cast<jlong>(archive.Data().Size()));

        NJava::TMetricArchiveClass metricArchiveClass{jenv};
        return metricArchiveClass.New(
                javaHeader,
                javaFormat,
                static_cast<jint>(archive.Columns().Mask()),
                bitBuf,
                static_cast<jint>(archive.PointCount()));
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        jenv->ThrowNew(jenv->FindClass("java/lang/RuntimeException"), msg.c_str());
        return nullptr;
    }
}

jbyteArray Java_ru_yandex_solomon_MetricArchiveNative_encode(JNIEnv* jenv, jclass, jobject javaArchiveObject) {
    try {
        NJava::TMetricArchiveClass metricArchiveClass{jenv};
        NJava::TMetricArchiveObject javaArchive = metricArchiveClass.Convert(javaArchiveObject);

        NJava::TStockpileFormatClass formatClass{jenv};
        EFormat format = static_cast<EFormat>(formatClass.Convert(javaArchive.Format));

        NJava::TMetricTypeClass metricTypeClass{jenv};
        jint typeNumber = metricTypeClass.GetNumber(javaArchive.Header.Type);

        TMetricHeader header;
        header.DeleteBefore = TInstant::MilliSeconds(static_cast<ui64>(javaArchive.Header.DeleteBefore));
        header.Owner.ProjectId = static_cast<ui32>(javaArchive.Header.ProjectId);
        header.Owner.ShardId = static_cast<ui32>(javaArchive.Header.ShardId);
        header.DecimPolicyId = static_cast<ui32>(javaArchive.Header.DecimPolicyId);
        header.Type = static_cast<yandex::solomon::model::MetricType>(typeNumber);

        NTs::TColumnSet columns{static_cast<NTs::TColumnsMask>(javaArchive.ColumnSetMask)};
        ui32 pointCount = static_cast<ui32>(javaArchive.RecordCount);

        NJava::THeapBitBufClass heapBitBufClass{jenv};
        NJava::THeapBitBufObject bitBuf = heapBitBufClass.Convert(javaArchive.Buffer);

        TBuffer buf = NJava::ToBuffer(jenv, bitBuf.Array);
        ui64 dataBitSize = static_cast<ui64>(bitBuf.WriteIndex - bitBuf.ReadIndex);

        NTs::TBitBuffer data(std::move(buf));
        data.Resize(dataBitSize);
        TMetricArchive archive{header, format, columns, pointCount, std::move(data)};

        return NJava::ToByteArray(jenv, Encode(archive));
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        jenv->ThrowNew(jenv->FindClass("java/lang/RuntimeException"), msg.c_str());
        return nullptr;
    }
}
