package ru.yandex.solomon;

import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Sergey Polovko
 */
@RunWith(Parameterized.class)
public class MetricArchiveTest {

    @Parameterized.Parameter
    public StockpileFormat format;

    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return StockpileFormat.values();
    }

    @Test
    public void decode() {
        var header = new MetricHeader(System.currentTimeMillis(), 11, 12, 13, MetricType.COUNTER);
        var archive = new MetricArchiveMutable(header, format);
        archive.addRecord(AggrPoints.lpoint("2020-05-09T10:00:00Z", 1));
        archive.addRecord(AggrPoints.lpoint("2020-05-09T10:00:15Z", 2));
        archive.addRecord(AggrPoints.lpoint("2020-05-09T10:00:30Z", 3));
        archive.addRecord(AggrPoints.lpoint("2020-05-09T10:00:45Z", 4));
        archive.addRecord(AggrPoints.lpoint("2020-05-09T10:01:00Z", 5));

        var expected = archive.toImmutable();
        var decoded = MetricArchiveNative.decode(format, serialize(expected));
        Assert.assertEquals(expected, decoded);
    }

    @Test
    public void encode() {
        var header = new MetricHeader(System.currentTimeMillis(), 21, 22, 23, MetricType.DGAUGE);
        var archive = new MetricArchiveMutable(header, format);
        archive.addRecord(AggrPoints.point("2020-05-09T10:00:00Z", 1.0));
        archive.addRecord(AggrPoints.point("2020-05-09T10:00:15Z", 2.0));
        archive.addRecord(AggrPoints.point("2020-05-09T10:00:30Z", 3.0));
        archive.addRecord(AggrPoints.point("2020-05-09T10:00:45Z", 4.0));
        archive.addRecord(AggrPoints.point("2020-05-09T10:01:00Z", 5.0));

        var expected = archive.toImmutable();

        var javaBuffer = serialize(expected);
        var cppBuffer = MetricArchiveNative.encode(expected);
        Assert.assertArrayEquals(javaBuffer, cppBuffer);

        var encoded = deserialize(format, cppBuffer);
        Assert.assertEquals(expected, encoded);
    }

    private static byte[] serialize(MetricArchiveImmutable ma) {
        var serializer = MetricArchiveNakedSerializer.serializerForFormatSealed(ma.getFormat());
        return serializer.serializeToBytes(ma);
    }

    private static MetricArchiveImmutable deserialize(StockpileFormat format, byte[] buffer) {
        var serializer = MetricArchiveNakedSerializer.serializerForFormatSealed(format);
        return serializer.deserializeFull(buffer);
    }
}
