#pragma once

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/generic/string.h>
#include <util/generic/strbuf.h>

#include <string_view>

namespace NSolomon {
namespace NPrivate {

// Special hash functor for heterogeneous lookups.
// Read https://abseil.io/tips/144 for more details.
struct TStrHash {
    // A transparent functor is one that accepts more than one particular type.
    // It must publicize this fact by providing an is_transparent inner type
    using is_transparent = void;

    size_t operator()(const TString& s) const noexcept {
        return NHashPrivate::ComputeStringHash(s.data(), s.size());
    }

    size_t operator()(TStringBuf s) const noexcept {
        return NHashPrivate::ComputeStringHash(s.data(), s.size());
    }

    size_t operator()(std::string_view s) const noexcept {
        return NHashPrivate::ComputeStringHash(s.data(), s.size());
    }
};

// Special compare functor for heterogeneous lookups.
// Read https://abseil.io/tips/144 for more details.
struct TStrEq {
    // A transparent functor is one that accepts more than one particular type.
    // It must publicize this fact by providing an is_transparent inner type
    using is_transparent = void;

    bool operator()(TStringBuf l, TStringBuf r) const noexcept {
        return l == r;
    }

    bool operator()(TStringBuf l, const TString& r) const noexcept {
        return l == r;
    }

    bool operator()(const TString& l, TStringBuf r) const noexcept {
        return l == r;
    }

    bool operator()(std::string_view l, std::string_view r) const noexcept {
        return l == r;
    }

    bool operator()(std::string_view l, const TString& r) const noexcept {
        return l == r;
    }

    bool operator()(const TString& l, std::string_view r) const noexcept {
        return l == r;
    }

    bool operator()(const TString& l, const TString& r) const noexcept {
        return l == r;
    }
};

} // namespace NPrivate

template <typename T>
using TStringMap = absl::flat_hash_map<TString, T, NPrivate::TStrHash, NPrivate::TStrEq>;

} // namespace NSolomon
