#pragma once

#include <solomon/protos/common/string_pool.pb.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/generic/buffer.h>
#include <util/generic/noncopyable.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>

namespace NSolomon::NStringPool {

class TStringPoolBuilder;

/**
 * Pool with uniques strings. Allows to access string reference by its unique ID.
 * All strings are null-terminated
 */
class TStringPool: private TMoveOnly {
    friend TStringPoolBuilder;
public:
    TStringPool();

    /**
     * Restore pool from protobuf message. Will decompress string and restore index.
     *
     * @param pool  protobuf message with string pool
     */
    explicit TStringPool(const yandex::solomon::common::StringPool& pool);

    /**
     * Retrieves string reference by its unique ID.
     *
     * @param string ID
     * @return string reference
     */
    TStringBuf operator[](ui32 id) const noexcept {
        return Index_[static_cast<size_t>(id)];
    }

    /**
     * Retrieves string reference by its unique ID.
     * Throws TOutOfRange exception if id is out of range
     *
     * @param string ID
     * @return string reference
     */
    TStringBuf At(ui32 id) const {
        Y_ENSURE(id < Index_.size(), "id #" << id << " is out of range string pool (" << Index_.size() << ").");
        return Index_[static_cast<size_t>(id)];
    }

    /**
     * @return size of underlying buffer
     */
    size_t SizeBytes() const noexcept {
        return Buf_.Size();
    }

    /**
     * @return number of strings stored in this pool
     */
    size_t Size() const noexcept {
        return Index_.size() - 1; // -1 for an empty string
    }

    /**
     * @return full copy of string pool
     */
     TStringPool Copy() const;

    /**
     * Compress undelrying strings and creates StringPool protobuf message with them.
     *
     * @param c compression algorithm
     * @return protobuf message with string pool
     */
    yandex::solomon::common::StringPool ToProto(yandex::solomon::common::StringPool::Compression c) const;

    /**
     * Compress undelrying strings and creates StringPool protobuf message with them.
     * Same as above method but more efficient when protobuf message is allocated on arena.
     *
     * @param c    compression algorithm
     * @param pool mutable protobuf message with string pool
     */
    void ToProto(
            yandex::solomon::common::StringPool::Compression c,
            yandex::solomon::common::StringPool* pool) const;

private:
    explicit TStringPool(TBuffer buf);

private:
    TBuffer Buf_;
    TVector<TStringBuf> Index_; // id -> string
};

/**
 * Builder of string pool. Uses single chunk of memory to store all strings delimited by '\0'.
 * In hash map strings are stored as (offset, size) to reference data actually stored in flat buffer.
 */
class TStringPoolBuilder: private TMoveOnly {
    /**
     * Reference to a string stored in the pool buffer
     */
    struct TStringRef {
        ui32 Offset;
        ui32 Size;
    };

    /**
     * Hasher for TStringRef and TStringBuf
     *
     * Special hash functor for heterogeneous lookups.
     * Read https://abseil.io/tips/144 for more details.
     */
    struct TStringRefHash {
        // A transparent functor is one that accepts more than one particular type.
        // It must publicize this fact by providing an is_transparent inner type
        using is_transparent = void;

        const TBuffer* Buffer;

        ui64 operator()(const TStringRef& s) const noexcept {
            return CityHash64(Buffer->Data() + s.Offset, static_cast<size_t>(s.Size));
        }

        ui64 operator()(TStringBuf s) const noexcept {
            return CityHash64(s.data(), s.size());
        }
    };

    /**
     * Equal implementation for TStringRef and TStringBuf
     *
     * Special hash functor for heterogeneous lookups.
     * Read https://abseil.io/tips/144 for more details.
     */
    struct TStringRefEq {
        // A transparent functor is one that accepts more than one particular type.
        // It must publicize this fact by providing an is_transparent inner type
        using is_transparent = void;

        const TBuffer* Buffer;

        bool operator()(const TStringRef& a, const TStringRef& b) const noexcept {
            return a.Offset == b.Offset && a.Size == b.Size;
        }

        bool operator()(TStringRef a, TStringBuf b) const noexcept {
            TStringBuf c{Buffer->Data() + a.Offset, static_cast<size_t>(a.Size)};
            return c == b;
        }
    };

public:
    /**
     * Create pool builder with given buffer initial capacity.
     *
     * @param capacityBytes  initial capacity for underlying buffer
     */
    explicit TStringPoolBuilder(size_t capacityBytes = 128);

    /**
     * Put string in pool if it was absent and return its unique ID.
     * If the string previously was added to this pool builder then just return it ID.
     *
     * @param value  string to add
     * @return unique ID of string in this pool
     */
    ui32 Put(TStringBuf value);

    /**
     * @return full copy of string pool builder
     */
    TStringPoolBuilder Copy() const;

    /**
     * Creates string pool from underline buffer.
     * String pool builder will be cleaned after this member-function call.
     *
     * @return string pool
     */
    TStringPool Build();

    /**
     * Compress strings stored in underlying buffer and creates protobuf message.
     *
     * @param c  compression algorithm
     * @return protobuf message with string pool
     */
    yandex::solomon::common::StringPool Build(yandex::solomon::common::StringPool::Compression c) const;

    /**
     * @return a view of underlying buffer stored by this builder.
     */
    TStringBuf UnderlyingBuf() const {
        return {Buf_->Data(), Buf_->Size()};
    }

    /**
     * @return current size of underlying buffer
     */
    size_t SizeBytes() const noexcept {
        return Buf_->Size();
    }

    /**
     * @return number of strings stored in this pool
     */
    size_t Size() const noexcept {
        return Index_.size();
    }

private:
    std::unique_ptr<TBuffer> Buf_;
    absl::flat_hash_map<TStringRef, ui32, TStringRefHash, TStringRefEq> Index_; // string -> id
};

} // namespace NSolomon::NStringPool
