#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>

using namespace NSolomon::NStringPool;
using namespace yandex::solomon::common;

TEST(TStringPoolTest, BuilderPut) {
    TStringPoolBuilder b;
    EXPECT_EQ(b.Size(), 0u);
    EXPECT_EQ(b.SizeBytes(), 1u);

    ui32 solomonId = b.Put("solomon");
    EXPECT_EQ(solomonId, 1u);
    EXPECT_EQ(b.Size(), 1u);
    EXPECT_EQ(b.SizeBytes(), 9u);

    ui32 graphiteId = b.Put("graphite");
    EXPECT_EQ(graphiteId, 2u);
    EXPECT_EQ(b.Size(), 2u);
    EXPECT_EQ(b.SizeBytes(), 18u);

    ui32 emptyId = b.Put("");
    EXPECT_EQ(emptyId, 0u);
    EXPECT_EQ(b.Size(), 2u);
    EXPECT_EQ(b.SizeBytes(), 18u);

    // put the same values

    EXPECT_EQ(b.Put("graphite"), 2u);
    EXPECT_EQ(b.Size(), 2u);
    EXPECT_EQ(b.SizeBytes(), 18u);

    EXPECT_EQ(b.Put("solomon"), 1u);
    EXPECT_EQ(b.Size(), 2u);
    EXPECT_EQ(b.SizeBytes(), 18u);

    EXPECT_EQ(b.Put(""), 0u);
    EXPECT_EQ(b.Size(), 2u);
    EXPECT_EQ(b.SizeBytes(), 18u);

    //  put many unique values

    THashMap<TString, ui32> ids;
    for (size_t i = 0; i < 1000; i++) {
        TString s = TStringBuf("s") + ToString(i);
        ui32 id = b.Put(s);
        ids.emplace(s, id);
    }

    size_t prevSize = b.Size();
    size_t prevSizeBytes = b.SizeBytes();

    for (const auto& [s, id]: ids) {
        EXPECT_EQ(b.Put(s), id);
    }

    EXPECT_EQ(b.Size(), prevSize);
    EXPECT_EQ(b.SizeBytes(), prevSizeBytes);
}

TEST(TStringPoolTest, BuilderBuildPool) {
    TStringPoolBuilder b;
    ui32 aId = b.Put("aaaaa");
    ui32 bId = b.Put("bbbbb");
    ui32 cId = b.Put("ccccc");
    EXPECT_EQ(b.Size(), 3u);
    EXPECT_EQ(b.SizeBytes(), 19u);

    {
        TStringPool pool = b.Build();
        EXPECT_EQ(pool.Size(), 3u);
        EXPECT_EQ(pool.SizeBytes(), 19u);
        EXPECT_EQ(pool[aId], "aaaaa");
        EXPECT_EQ(pool[bId], "bbbbb");
        EXPECT_EQ(pool[cId], "ccccc");
    }

    // builder was cleaned
    EXPECT_EQ(b.Size(), 0u);
    EXPECT_EQ(b.SizeBytes(), 0u);

    ui32 dId = b.Put("ddddd");
    ui32 eId = b.Put("eeeee");
    EXPECT_EQ(b.Size(), 2u);
    EXPECT_EQ(b.SizeBytes(), 12u);

    {
        TStringPool pool = b.Build();
        EXPECT_EQ(pool.Size(), 2u);
        EXPECT_EQ(pool.SizeBytes(), 12u);
        EXPECT_EQ(pool[dId], "ddddd");
        EXPECT_EQ(pool[eId], "eeeee");
    }
}

TEST(TStringPoolTest, BuilderBuildMessage) {
    TStringPoolBuilder b;
    b.Put("aaaaa");
    b.Put("bbbbb");
    b.Put("ccccc");
    b.Put("ddddd");
    EXPECT_EQ(b.Size(), 4u);
    EXPECT_EQ(b.SizeBytes(), 25u);

    // no compression
    {
        auto pool = b.Build(StringPool_Compression_COMPRESSION_UNSPECIFIED);
        EXPECT_EQ(pool.compression(), StringPool_Compression_COMPRESSION_UNSPECIFIED);
        EXPECT_EQ(pool.original_size(), 25u);

        TStringBuf buf{"\0aaaaa\0bbbbb\0ccccc\0ddddd\0", 25};
        EXPECT_EQ(pool.strings(), buf);
    }

    // lz4 compression
    {
        auto pool = b.Build(StringPool_Compression_LZ4);
        EXPECT_EQ(pool.compression(), StringPool_Compression_LZ4);
        EXPECT_EQ(pool.original_size(), 25u);
        EXPECT_LT(pool.strings().size(), 25u);
    }
}

TEST(TStringPoolTest, BuildRestore) {
    TStringPoolBuilder b;
    b.Put("aaaaa");
    b.Put("bbbbb");
    b.Put("ccccc");
    b.Put("ddddd");
    EXPECT_EQ(b.Size(), 4u);
    EXPECT_EQ(b.SizeBytes(), 25u);

    for (int c = StringPool_Compression_Compression_MIN; c < StringPool_Compression_Compression_MAX; c++) {
        auto msg = b.Build(static_cast<StringPool::Compression>(c));
        EXPECT_EQ(msg.compression(), c);
        EXPECT_EQ(msg.original_size(), 25u);
        EXPECT_FALSE(msg.strings().empty());

        TStringPool pool(msg);
        EXPECT_EQ(pool.Size(), 4u);
        EXPECT_EQ(pool.SizeBytes(), 25u);

        EXPECT_EQ("aaaaa", pool[1]);
        EXPECT_EQ("bbbbb", pool[2]);
        EXPECT_EQ("ccccc", pool[3]);
        EXPECT_EQ("ddddd", pool[4]);
    }
}

TEST(TStringPoolTest, Copy) {
    TStringPool copy;
    {
        TStringPoolBuilder sb;
        sb.Put("aaaaa");
        sb.Put("bbbbb");
        sb.Put("ccccc");
        sb.Put("ddddd");

        TStringPool origin = sb.Build();
        copy = origin.Copy();
    }

    EXPECT_EQ(copy.Size(), 4u);
    EXPECT_EQ(copy.SizeBytes(), 25u);

    EXPECT_EQ(copy[1], "aaaaa");
    EXPECT_EQ(copy[2], "bbbbb");
    EXPECT_EQ(copy[3], "ccccc");
    EXPECT_EQ(copy[4], "ddddd");
}

TEST(TStringPoolTest, At) {
    TStringPoolBuilder poolBuilder;
    ui32 aId = poolBuilder.Put("aaaaa");
    ui32 bId = poolBuilder.Put("bbbbb");
    ui32 cId = poolBuilder.Put("ccccc");
    ui32 dId = poolBuilder.Put("ddddd");

    TStringPool pool = poolBuilder.Build();

    EXPECT_EQ(pool.Size(), 4u);

    EXPECT_EQ(pool.At(aId), "aaaaa");
    EXPECT_EQ(pool.At(bId), "bbbbb");
    EXPECT_EQ(pool.At(cId), "ccccc");
    EXPECT_EQ(pool.At(dId), "ddddd");

    EXPECT_ANY_THROW(pool.At(dId + 1));
    EXPECT_ANY_THROW(pool.At(1247329934u));
    EXPECT_ANY_THROW(pool.At(Max<ui32>()));
}

TEST(TStringPoolTest, StringPoolIsMovable) {
    TStringPoolBuilder origin;

    auto aId = origin.Put("aaaaa");
    auto bId = origin.Put("bbbbb");

    auto size = origin.Size();
    auto sizeBytes = origin.SizeBytes();

    auto moved = std::move(origin);
    EXPECT_EQ(moved.Size(), size);
    EXPECT_EQ(moved.SizeBytes(), sizeBytes);

    auto cId = moved.Put("ccccc");

    auto pool = moved.Build();
    EXPECT_EQ(pool.At(aId), "aaaaa");
    EXPECT_EQ(pool.At(bId), "bbbbb");
    EXPECT_EQ(pool.At(cId), "ccccc");
}
