#pragma once

#include "access.h"

#include <util/system/mutex.h>
#include <util/system/spinlock.h>

namespace NSolomon::NSync {

/**
 * Rust like a mutual exclusion lock.
 *
 * @tparam T           value type
 * @tparam TLockImpl   lock type
 */
template <typename T, typename TLockImpl = ::TMutex>
class TLock: public TNonCopyable {
public:
    struct TAccess: public TAccessMut<T, TGuard<TLockImpl>> {
        TAccess(T* data, TLockImpl* lock) noexcept
            : TAccessMut<T, TGuard<TLockImpl>>{data, lock}
        {
        }
    };

    struct TTryAccess: public TAccessMut<T, TTryGuard<TLockImpl>> {
        TTryAccess(T* data, TLockImpl* lock) noexcept
            : TAccessMut<T, TTryGuard<TLockImpl>>{data, lock}
        {
        }
    };

public:
    TLock() noexcept(std::is_nothrow_default_constructible_v<T>) = default;

    /**
     * Creates an instance of the value and associated mutex in an unlocked state.
     *
     * @param data  initial value
     */
    template <typename... TArgs>
    TLock(TArgs&&... args) noexcept (std::is_nothrow_constructible_v<T, TArgs...>)
        : Data_{std::forward<TArgs>(args)...}
    {
    }

    /**
     * Acquires a mutex, blocking the current thread until it is able to do so.
     * This function will block the local thread until it is available to acquire
     * the mutex.
     *
     * @return an RAII guard which will release underlying lock on destruction.
     */
    TAccess Lock() {
        return {&Data_, &Lock_};
    }

    /**
     * Attempts to acquire this lock.
     * This function does not block current thread.
     *
     * @return an RAII guard which will release underlying lock on destruction iff it was acquired.
     */
    TTryAccess TryLock() {
        return {&Data_, &Lock_};
    }

private:
    T Data_;
    mutable TLockImpl Lock_;
};

template <typename T>
using TAdaptiveLock = ::NSolomon::NSync::TLock<T, ::TAdaptiveLock>;

} // namespace NSolomon::NSync
