#pragma once

#include "access.h"

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

namespace NSolomon::NSync {

/**
 * Rust like read-write lock.
 *
 * @tparam T           value type
 * @tparam TLockImpl   lock type
 */
template <typename T, typename TLockImpl = ::TRWMutex>
class TRwLock: public TNonCopyable {
public:
    struct TRead: public TAccessConst<T, TReadGuardBase<TLockImpl>> {
        TRead(const T* data, TLockImpl* lock) noexcept
            : TAccessConst<T, TReadGuardBase<TLockImpl>>{data, lock}
        {
        }
    };

    struct TTryRead: public TAccessConst<T, TTryReadGuardBase<TLockImpl>> {
        TTryRead(const T* data, TLockImpl* lock) noexcept
            : TAccessConst<T, TTryReadGuardBase<TLockImpl>>{data, lock}
        {
        }
    };

    struct TWrite: public TAccessMut<T, TWriteGuardBase<TLockImpl>> {
        TWrite(T* data, TLockImpl* lock) noexcept
            : TAccessMut<T, TWriteGuardBase<TLockImpl>>{data, lock}
        {
        }
    };

    struct TTryWrite: public TAccessMut<T, TTryWriteGuardBase<TLockImpl>> {
        TTryWrite(T* data, TLockImpl* lock) noexcept
            : TAccessMut<T, TTryWriteGuardBase<TLockImpl>>{data, lock}
        {
        }
    };

public:
    /**
     * Creates default instance of the value and associated read-write lock, which is unlocked.
     */
    TRwLock() noexcept(std::is_nothrow_default_constructible_v<T>) = default;

    /**
     * Creates an instance of the value and associated read-write lock, which is unlocked.
     *
     * @param data  initial value
     */
    template <typename... TArgs>
    TRwLock(TArgs&&... args) noexcept (std::is_nothrow_constructible_v<T, TArgs...>)
        : Data_{std::forward<TArgs>(args)...}
    {
    }

    /**
     * Locks this read-write lock with shared read access, blocking the current thread
     * until it can be acquired. The calling thread will be blocked until there are no
     * more writers which hold the lock.
     *
     * @return an RAII guard which will release underlying read lock on destruction.
     */
    TRead Read() const {
        return {&Data_, &Lock_};
    }

    /**
     * Attempts to acquire this read-write lock with shared read access.
     * This function does not block current thread.
     *
     * @return an RAII guard which will release underlying read lock on destruction iff it was acquired.
     */
    TTryRead TryRead() const {
        return {&Data_, &Lock_};
    }

    /**
     * Locks this read-write with exclusive write access, blocking the current thread
     * until it can be acquired. This function will not return while other writers or other readers
     * currently have access to the lock.
     *
     * @return an RAII guard which will release underlying write lock on destruction.
     */
    TWrite Write() {
        return {&Data_, &Lock_};
    }

    /**
     * Attempts to lock this read-write with exclusive write access.
     * This function does not block current thread.
     *
     * @return an RAII guard which will release underlying write lock on destruction iff it was acquired.
     */
    TTryWrite TryWrite() {
        return {&Data_, &Lock_};
    }

private:
    T Data_;
    mutable TLockImpl Lock_;
};

template <typename T>
using TLightRwLock = ::NSolomon::NSync::TRwLock<T, ::TLightRWLock>;

} // namespace NSolomon::NSync
