#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;

TEST(TRwLockTest, Read) {
    NSync::TRwLock<int> val = 42;

    auto v = val.Read();
    ASSERT_TRUE(v);
    ASSERT_EQ(*v, 42);
}

TEST(TRwLockTest, Write) {
    NSync::TRwLock<int> val = 42;

    {
        auto v = val.Write();
        ASSERT_TRUE(v);
        ASSERT_EQ(*v, 42);

        *v = 57;
        ASSERT_EQ(*v, 57);
    }

    auto v = val.Read();
    ASSERT_TRUE(v);
    ASSERT_EQ(*v, 57);
}

TEST(TRwLockTest, TryRead) {
    NSync::TRwLock<int> val = 42;

    {
        auto v1 = val.TryRead();
        ASSERT_TRUE(v1);
        ASSERT_EQ(*v1, 42);

        // several readers can access the value
        auto v2 = val.TryRead();
        ASSERT_TRUE(v2);
        ASSERT_EQ(*v2, 42);
    }

    {
        auto v1 = val.Write();
        ASSERT_TRUE(v1);
        ASSERT_EQ(*v1, 42);

        // there is the active writer, so reader cannot access value
        auto v2 = val.TryRead();
        ASSERT_FALSE(v2);
    }

    // writer is gone, so reader can read value
    auto v = val.TryRead();
    ASSERT_TRUE(v);
    ASSERT_EQ(*v, 42);
}

TEST(TRwLockTest, TryWrite) {
    NSync::TRwLock<int> val = 42;

    {
        auto v1 = val.TryWrite();
        ASSERT_TRUE(v1);
        ASSERT_EQ(*v1, 42);

        // only one writer can access the value
        auto v2 = val.TryWrite();
        ASSERT_FALSE(v2);
    }

    {
        auto v1 = val.Read();
        ASSERT_TRUE(v1);
        ASSERT_EQ(*v1, 42);

        // there is the active reader, so writer cannot access the value
        auto v2 = val.TryWrite();
        ASSERT_FALSE(v2);
    }

    // reader is gone, so writer can access value
    auto v = val.TryWrite();
    ASSERT_TRUE(v);
    ASSERT_EQ(*v, 42);

    *v = 57;
    ASSERT_EQ(*v, 57);
}

TEST(TRwLockTest, ComplexTypes) {
    {
        NSync::TRwLock<TString> str = "initialized from char*";
        ASSERT_EQ(*str.Read(), "initialized from char*");
    }
    {
        NSync::TRwLock<TString> str = TStringBuf{"initialized from string buf"};
        ASSERT_EQ(*str.Read(), "initialized from string buf");
    }
    {
        NSync::TRwLock<TString> str = TString{"initialized from string"};
        ASSERT_EQ(*str.Read(), "initialized from string");
    }
    {
        using TMyMap = std::map<int, TString>;
        NSync::TRwLock<TMyMap> map = TMyMap{
                {1, "one"},
                {2, "two"},
                {3, "three"},
        };

        if (auto r = map.Read()) {
            ASSERT_EQ(r->size(), 3u);
            ASSERT_EQ(r->at(1), "one");

            auto it = r->find(2);
            ASSERT_NE(it, r->end());
            ASSERT_EQ(it->second, "two");
        } else {
            FAIL() << "failed to acquire read lock";
        }

        if (auto w = map.Write()) {
            ASSERT_EQ(w->size(), 3u);
            ASSERT_EQ(w->at(1), "one");

            (*w)[4] = "four";
            ASSERT_EQ(w->at(4), "four");

            ASSERT_EQ(w->erase(3), 1u);
            ASSERT_EQ(w->find(3), w->end());
        } else {
            FAIL() << "failed to acquire write lock";
        }
    }
}

TEST(TRwLockTest, DefaultConstructable) {
    NSync::TRwLock<TInstant> val;

    auto v = val.Read();
    ASSERT_TRUE(v);
    ASSERT_EQ(*v, TInstant::Zero());
}
