#include "proc.h"

#include <contrib/libs/re2/re2/re2.h>

#include <util/system/platform.h>
#include <util/system/file.h>
#include <util/system/spinlock.h>
#include <util/system/guard.h>

#include <util/system/yassert.h>

#include <util/stream/file.h>

#include <util/generic/vector.h>
#include <util/generic/set.h>
#include <util/generic/hash.h>

#include <util/string/split.h>

#include <util/folder/path.h>

#ifdef _unix_
#include <sys/stat.h>
#include <sys/vfs.h>
#endif

#ifdef _win_
#include <Windows.h>
#endif

namespace NSolomon {
namespace {
    TAdaptiveLock SvctmLock;

    struct TSvctm {
        ui64 Reads;
        ui64 Writes;
        ui64 IoTime;
    };

    void ProcUptime(Sysmon* r, const TFsPath& procDirectory) {
        try {
            TFileInput is(procDirectory / "uptime");
            TString line = is.ReadLine();
            TVector<TStringBuf> fields;
            Split(line.data(), " ", fields);
            if (fields.size() < 1) {
                return;
            }
            ui64 uptime = (ui64)FromString<double>(fields[0]) * 1000;
            r->MutableSystem()->SetUpTime(uptime);
            r->MutableSystem()->SetUpTimeRaw(uptime);
        } catch (...) {
        }
    }

    float TICKS_PER_MILLISEC =
#ifdef _SC_CLK_TCK
        sysconf(_SC_CLK_TCK) / 1000.0;
#else
        1.f;
#endif

    float TICK_TO_MILLISEC = 1.0 / TICKS_PER_MILLISEC;

    template <typename TVal>
    static bool ExtractValue(const TString& str, const TString& name, TVal* res) {
        if (!str.StartsWith(name))
            return false;
        size_t pos = name.size();
        while (pos < str.size() && (str[pos] == ' ' || str[pos] == '\t')) {
            pos++;
        }
        size_t pos2 = pos;
        while (pos2 < str.size() && (str[pos2] != ' ' && str[pos2] != '\t')) {
            pos2++;
        }
        try {
            *res = FromString(TStringBuf(str.data() + pos, str.data() + pos2));
            return true;
        } catch (...) {
            return false;
        }
    }

    void ExtractStatCpuInfo(const TString& line, Sysmon* r) {
        TVector<TStringBuf> fields;
        Split(line.data(), " ", fields);

        if (fields.size() < 8) {
            return;
        }

        SysmonSystem* node;

        if ("cpu" == fields[0]) {
            node = r->MutableSystem();
        } else {
            i64 cpuId;

            if (!TryFromString(fields[0].Skip(3), cpuId)) {
                return;
            }

            node = r->MutableSystemCpu()->AddNode();
            node->SetCpuId(cpuId);
        }

        node->SetUserTime(FromString<ui64>(fields[1]) * TICK_TO_MILLISEC);
        node->SetNiceTime(FromString<ui64>(fields[2]) * TICK_TO_MILLISEC);
        node->SetSystemTime(FromString<ui64>(fields[3]) * TICK_TO_MILLISEC);
        node->SetIdleTime(FromString<ui64>(fields[4]) * TICK_TO_MILLISEC);
        node->SetIoWaitTime(FromString<ui64>(fields[5]) * TICK_TO_MILLISEC);
        node->SetIrqTime(FromString<ui64>(fields[6]) * TICK_TO_MILLISEC + FromString<ui64>(fields[7]) * TICK_TO_MILLISEC);

        if (fields.size() >= 9) {
            node->SetStealTime(FromString<ui64>(fields[8]) * TICK_TO_MILLISEC);
        }
        if (fields.size() >= 10) {
            node->SetGuestTime(FromString<ui64>(fields[9]) * TICK_TO_MILLISEC);
        }
        if (fields.size() >= 11) {
            node->SetGuestNiceTime(FromString<ui64>(fields[10]) * TICK_TO_MILLISEC);
        }
    }

    void ProcLoadavg(Sysmon* r, const TFsPath& procDirectory) {
        try {
            TFileInput is(procDirectory / "loadavg");

            TString firstLine = is.ReadLine();
            TVector<TStringBuf> fields;
            Split(firstLine.data(), " ", fields);

            if (fields.size() < 5) {
                return;
            }

            r->MutableProc()->SetLoadAverage1min(FromString(fields[0]));
            r->MutableProc()->SetLoadAverage5min(FromString(fields[1]));
            r->MutableProc()->SetLoadAverage15min(FromString(fields[2]));

            TVector<TStringBuf> parts;
            Split(fields[3], "/", parts);

            if (parts.size() != 2) {
                return;
            }

            r->MutableProc()->SetThreads(FromString(parts[1]));
            r->MutableProc()->SetLa(FromString(fields[1]));
        } catch (...) {
        }
    }

#define READ_MEM_INFO(column, setter)             \
    if (ExtractValue(line, column, &value)) {     \
        r->MutableMemory()->setter(value * 1024); \
        continue;                                 \
    }

    void ProcMeminfo(Sysmon* r, const TFsPath& procDirectory) {
        try {
            TFileInput is(procDirectory / "meminfo");

            TString line;
            while (is.ReadLine(line)) {
                ui64 value;
                READ_MEM_INFO("MemTotal:", SetMemTotal)
                READ_MEM_INFO("MemFree:", SetMemFree)
                READ_MEM_INFO("MemAvailable:", SetMemAvailable)
                READ_MEM_INFO("Buffers:", SetBuffers)
                READ_MEM_INFO("Cached:", SetCached)
                READ_MEM_INFO("SwapCached:", SetSwapCached)
                READ_MEM_INFO("Active:", SetActive)
                READ_MEM_INFO("Inactive:", SetInactive)
                READ_MEM_INFO("Active(anon):", SetActiveAnon)
                READ_MEM_INFO("Inactive(anon):", SetInactiveAnon)
                READ_MEM_INFO("Active(file):", SetActiveFile)
                READ_MEM_INFO("Inactive(file):", SetInactiveFile)
                READ_MEM_INFO("Unevictable:", SetUnevictable)
                READ_MEM_INFO("Mlocked:", SetMlocked)
                READ_MEM_INFO("SwapTotal:", SetSwapTotal)
                READ_MEM_INFO("SwapFree:", SetSwapFree)
                READ_MEM_INFO("Dirty:", SetDirty)
                READ_MEM_INFO("Writeback:", SetWriteback)
                READ_MEM_INFO("AnonPages:", SetAnonPages)
                READ_MEM_INFO("Mapped:", SetMapped)
                READ_MEM_INFO("Shmem:", SetShmem)
                READ_MEM_INFO("Committed_AS:", SetCommitted_AS)
                READ_MEM_INFO("CommitLimit:", SetCommitLimit)
                READ_MEM_INFO("Slab:", SetSlab)
                READ_MEM_INFO("SReclaimable:", SetSReclaimable)
                READ_MEM_INFO("SUnreclaim:", SetSUnreclaim)
                READ_MEM_INFO("AnonHugePages:", SetAnonHugePages)
                READ_MEM_INFO("PageTables:", SetPageTables)
            }
        } catch (...) {
        }
    }

    void ProcVmstat(Sysmon* r, const TFsPath& procDirectory) {
        try {
            TFileInput proc(procDirectory / "vmstat");
            TString str;
            while (proc.ReadLine(str)) {
                ui64 value;
                if (ExtractValue(str, "pgmajfault", &value)) {
                    r->MutableMemory()->SetMajorPageFaults(value);
                }
                if (ExtractValue(str, "pgpgin", &value)) {
                    r->MutableMemory()->SetPageIns(value);
                }
                if (ExtractValue(str, "pgpgout", &value)) {
                    r->MutableMemory()->SetPageOuts(value);
                }
                if (ExtractValue(str, "pswpin", &value)) {
                    r->MutableMemory()->SetSwapIns(value);
                }
                if (ExtractValue(str, "pswpout", &value)) {
                    r->MutableMemory()->SetSwapOuts(value);
                }
            }
        } catch (...) {
        }
    }

    bool IsDiskGood(TStringBuf dev, const re2::RE2& nvmeNamePattern) {
        if (dev.StartsWith("md")) {
            return true;
        }
        // shelves can have drives with four letters (e.g. sdac)
        if (dev.StartsWith("sd")) {
            return !isdigit(dev.back());
        }
        // OpenStack virtual disk
        if (dev.StartsWith("vd")) {
            return dev.size() == 3;
        }
        // nvme0     - NVMe controller
        // nvme0n1   - NVMe disk on first NVMe controller
        // nvme1c1n1 - Also valid
        // nvme0n1p1 - First partition
        if (dev.StartsWith("nvme")) {
            return RE2::FullMatch(dev, nvmeNamePattern);
        }
        return false;
    }

    double CalcSvctm(const TString &dev, ui64 reads, ui64 writes, ui64 iotime) {
        with_lock(SvctmLock) {
            static THashMap<TString, TSvctm> svctm;

            auto ptr = svctm.insert({dev, {reads, writes, iotime}});
            if (ptr.second) {
                return 0;
            }

            auto &it = ptr.first->second;
            if (iotime < it.IoTime || reads < it.Reads || writes < it.Writes) {
                it = {reads, writes, iotime};
                return 0;
            }

            ui64 iotimeDiff = iotime - it.IoTime;
            ui64 rwDiff = (reads - it.Reads) + (writes - it.Writes);
            double result = (rwDiff != 0) ? iotimeDiff / double(rwDiff) : iotimeDiff;
            it = {reads, writes, iotime};

            return result;
        }
    }

    struct TFsStat {
        TString Name;
        ui64 SpaceTotal{ 0 };
        ui64 SpaceAvailable{ 0 };
        ui64 SpaceUsed{ 0 };
        ui64 INodeTotal{ 0 };
        ui64 INodeFree{ 0 };
        ui64 INodeUsed{ 0 };
    };

#ifdef _unix_
    THashMap<ui64, TFsStat> GetFsInfo(const TFsPath& mountInfoPath) {
        THashMap<ui64, TFsStat> fsInfo;
        try {
            TFileInput is(mountInfoPath);

            TString line;
            struct statfs fsStats;
            struct stat stats;
            int res;

            while (is.ReadLine(line)) {
                TVector<TStringBuf> parts;
                Split(line, " ", parts);

                if (parts.size() < 10) {
                    continue;
                }

                TStringBuf name = parts[4];

                // this field actually has a filesystem-specific value, but in common
                // ones we can rely on it
                TStringBuf dev = parts[parts.size() - 2];

                if (!dev.StartsWith("/dev/")) {
                    continue;
                }

                const auto nameStr = TString{ name };
                res = statfs(nameStr.c_str(), &fsStats);
                if (res < 0) {
                    continue;
                }

                // get real device which the mountpoint belongs to
                res = stat(nameStr.c_str(), &stats);
                if (res < 0) {
                    continue;
                }

                TFsStat data{
                    TString{name},
                    fsStats.f_blocks * fsStats.f_bsize,
                    fsStats.f_bavail * fsStats.f_bsize,
                    (fsStats.f_blocks - fsStats.f_bfree) * fsStats.f_bsize,
                    fsStats.f_files,
                    fsStats.f_ffree,
                    fsStats.f_files - fsStats.f_ffree,
                };

                auto it = fsInfo.find(stats.st_dev);
                if (it == std::end(fsInfo)) {
                    fsInfo[stats.st_dev] = std::move(data);
                } else {
                    // trying to determine which one is closer to the root
                    // like df does
                    if (data.Name.size() < it->second.Name.size()) {
                        it->second = std::move(data);
                    }
                }
            }
        } catch (...) {
        }
        return fsInfo;
    }
#else
THashMap<ui64, TFsStat> GetFsInfo(const TFsPath&) {
    THashMap<ui64, TFsStat> fsInfo;

    constexpr ui64 BUFSIZE = 512;
    try {
        TCHAR szTemp[BUFSIZE];
        szTemp[0] = '\0';
        if (GetLogicalDriveStrings(BUFSIZE - 1, szTemp)) {
            TCHAR* szDrive = szTemp;
            ui64 count = 0;
            do {
                ULARGE_INTEGER p1, p2, p3;    // FreeBytesAvailableToCaller, TotalNumberOfBytes, TotalNumberOfFreeBytes
                GetDiskFreeSpaceEx(szDrive, &p1, &p2, &p3);

                TString name(szDrive);
                TFsStat data{
                    name,
                    p2.QuadPart,
                    p1.QuadPart,
                    p2.QuadPart - p1.QuadPart,
                    0,
                    0,
                    0
                };

                fsInfo[count++] = std::move(data);

                // Go to the next NULL character.
                while (*szDrive++);
            } while (*szDrive); // end of string
        }
    } catch (...) {
    }
    return fsInfo;
}
#endif

} // namespace

void ProcDiskstats(Sysmon* r, const TFsPath& procDirectory) {
    try {
        TFileInput is(procDirectory / "diskstats");
        static const re2::RE2 nvmeNamePattern{"nvme\\d+(c\\d+)?n\\d+"};
        TString line;
        while (is.ReadLine(line)) {
            TVector<TStringBuf> parts;
            Split(line, " ", parts);
            if (parts.size() < 14) {
                return;
            }

            TStringBuf dev = parts[2];
            if (!IsDiskGood(dev, nvmeNamePattern)) {
                continue;
            }

            SysmonDisk* disk = r->MutableIo()->AddDisks();
            disk->SetDev(TString(dev));
            disk->SetReads(FromString(parts[3]));
            disk->SetReadBytes(FromString<ui64>(parts[5]) * 512);
            disk->SetReadWaitMillisec(FromString<ui64>(parts[6]));
            disk->SetWrites(FromString(parts[7]));
            disk->SetWriteBytes(FromString<ui64>(parts[9]) * 512);
            disk->SetWriteWaitMillisec(FromString<ui64>(parts[10]));
            disk->SetIOsInProgress(FromString<ui64>(parts[11]));
            disk->SetIOMillisec(FromString<ui64>(parts[12]));
            disk->SetTimeInQueueMillisec(FromString<ui64>(parts[13]));
            disk->SetIOServiceTime(CalcSvctm(TString{dev},
                                             FromString<ui64>(parts[3]),
                                             FromString<ui64>(parts[7]),
                                             FromString<ui64>(parts[12])));

            if (disk->GetReads() > 0) {
                disk->SetReadWaitMillisecAvg(static_cast<double>(disk->GetReadWaitMillisec()) / disk->GetReads());
            } else {
                disk->SetReadWaitMillisecAvg(0);
            }

            if (disk->GetWrites() > 0) {
                disk->SetWriteWaitMillisecAvg(static_cast<double>(disk->GetWriteWaitMillisec()) / disk->GetWrites());
            } else {
                disk->SetWriteWaitMillisecAvg(0);
            }
        }
    } catch (...) {
    }
}


void ProcFilesystem(Sysmon* r, const TFsPath& mountInfoPath) {
    THashMap<ui64, TFsStat> processed = GetFsInfo(mountInfoPath);
    for (auto&& p: processed) {
        const auto& fs = p.second;

        SysmonMountpoint* disk = r->MutableFilesystem()->AddPoints();
        disk->SetDev(fs.Name);
        disk->SetSizeB(fs.SpaceTotal);
        disk->SetFreeB(fs.SpaceAvailable);
        disk->SetUsedB(fs.SpaceUsed);
        disk->SetINodeTotal(fs.INodeTotal);
        disk->SetINodeFree(fs.INodeFree);
        disk->SetINodeUsed(fs.INodeUsed);
    }
}

void ProcNuma(Sysmon *r, const TFsPath& sysDirectory) {
    TVector<TString> ls;
    try {
        TFsPath(sysDirectory / "devices/system/node").ListNames(ls);
    } catch (...) {
        return;
    }

    Numa *numa = nullptr;
    for (const auto& nodeName: ls) {
        try {
            TFileInput in(sysDirectory / "devices/system/node" / nodeName / "numastat");

            if (!numa) {
                numa = r->MutableNuma();
            }
            auto* node = numa->AddNode();
            node->SetName(nodeName);

            TString line;
            while (in.ReadLine(line)) {
                ui64 value;
                if (ExtractValue(line, "numa_hit", &value)) {
                    node->SetNumaHit(value);
                    continue;
                }
                if (ExtractValue(line, "numa_miss", &value)) {
                    node->SetNumaMiss(value);
                    continue;
                }
                if (ExtractValue(line, "numa_foreign", &value)) {
                    node->SetNumaForeign(value);
                    continue;
                }
                if (ExtractValue(line, "interleave_hit", &value)) {
                    node->SetInterleaveHit(value);
                    continue;
                }
                if (ExtractValue(line, "local_node", &value)) {
                    node->SetLocalNode(value);
                    continue;
                }
                if (ExtractValue(line, "other_node", &value)) {
                    node->SetOtherNode(value);
                    continue;
                }
            }
        } catch (...) {
        }
    }
}

void ProcStatSystem(Sysmon* r, const TFsPath& procDirectory) {
    try {
        TFileInput is(procDirectory / "stat");

        {
            TString firstLine = is.ReadLine();
            ExtractStatCpuInfo(firstLine, r);
        }

        TString line;
        while (is.ReadLine(line)) {
            ui64 value;
            if (ExtractValue(line, "procs_running", &value)) {
                r->MutableProc()->SetProcsRunning(value);
                continue;
            }
            if (ExtractValue(line, "procs_blocked", &value)) {
                r->MutableProc()->SetProcsBlocked(value);
                continue;
            }
            if (ExtractValue(line, "processes", &value)) {
                r->MutableProc()->SetProcsSinceBoot(value);
                continue;
            }
        }
    } catch (...) {
    }
}

void ProcStatCpu(Sysmon* r, const TFsPath& procDirectory) {
    try {
        TFileInput is(procDirectory / "stat");

        is.ReadLine(); // skip the first line

        TString line;
        while (is.ReadLine(line) && line.StartsWith("cpu")) {
            ExtractStatCpuInfo(line, r);
        }
    } catch (...) {
    }
}


void ProcMemory(Sysmon* sysmon, const TFsPath& procDirectory) {
    Y_ENSURE(sysmon != nullptr);
    ProcMeminfo(sysmon, procDirectory);
    ProcVmstat(sysmon, procDirectory);
}

void ProcKernel(Sysmon* sysmon, const TFsPath& procDirectory) {
    ProcUptime(sysmon, procDirectory);
    ProcStatSystem(sysmon, procDirectory);
    ProcLoadavg(sysmon, procDirectory);
}

Sysmon FillSysmon(const TFsPath& procDirectory, const TFsPath& sysDirectory) {
    Sysmon r;
    ProcUptime(&r, procDirectory);
    ProcStatSystem(&r, procDirectory);
    ProcStatCpu(&r, procDirectory);
    // TODO: total processes
    ProcLoadavg(&r, procDirectory);
    ProcMeminfo(&r, procDirectory);
    ProcVmstat(&r, procDirectory);
    ProcNet(r.MutableNet(), procDirectory);
    ProcDiskstats(&r, procDirectory);
    ProcCpuinfo(&r, procDirectory);
    ProcFilesystem(&r, procDirectory / "self/mountinfo");
    ProcNuma(&r, sysDirectory);
    return r;
}

} // namespace NSolomon
