#include "porto.h"

#include <solomon/libs/cpp/porto/porto.h>

#include <util/string/builder.h>
#include <util/generic/algorithm.h>

#include <contrib/libs/re2/re2/re2.h>

#include <array>

namespace NSolomon {
namespace {

TString MakeRegexp(const TVector<TString>& mappings) {
    TStringBuilder ss;

    ss << mappings[0];

    for (auto it = std::begin(mappings) + 1; it != std::end(mappings); ++it) {
        ss << "|" << *it;
    }

    return {ss};
}

TVector<NSolomon::TVolumeStats> GetVolumeStats(NSolomon::IPortoStatClient& portoClient, const TString& containerId, const TString& volumeNameRegex) {
    TVector<NSolomon::TVolumeStats> volumes;
    if (volumeNameRegex) {
        volumes = portoClient.VolumeStats(containerId);
        re2::RE2 regex{volumeNameRegex};
        Y_ENSURE(regex.ok(), "Cannot compile regex from string " << volumeNameRegex);
        EraseIf(volumes, [&] (auto&& volume) {
            return !RE2::FullMatch(volume.Path, regex);
        });
    }

    return volumes;
}

const TString ANY = ".*";

SysmonPorto StatsToProto(const TString& name, const NSolomon::TContainerStats& stats, const TVector<NSolomon::TVolumeStats>& volumes = {}) {
    SysmonPorto portoStat;

    portoStat.SetName(name);
    portoStat.SetCpuUsageMillis(stats.CpuUsageMillis);
    portoStat.SetCpuUsageSystemMillis(stats.CpuUsageSystemMillis);
    portoStat.SetCpuWaitMillis(stats.CpuWaitMillis);
    portoStat.SetCpuThrottledMillis(stats.CpuThrottledMillis);
    portoStat.SetCpuLimit(stats.CpuLimit);
    portoStat.SetCpuGuarantee(stats.CpuGuarantee);
    portoStat.SetCpuLimitTotal(stats.CpuLimitTotal);
    portoStat.SetCpuGuaranteeTotal(stats.CpuGuaranteeTotal);
    portoStat.SetMemoryUsageBytes(stats.MemoryUsageBytes);
    portoStat.SetAnonUsageBytes(stats.AnonUsageBytes);
    portoStat.SetAnonMaxUsageBytes(stats.AnonMaxUsageBytes);
    portoStat.SetCacheUsageBytes(stats.CacheUsageBytes);
    portoStat.SetMLockUsageBytes(stats.MLockUsageBytes);
    portoStat.SetMemoryLimit(stats.MemoryLimit);
    portoStat.SetMemoryLimitTotal(stats.MemoryLimitTotal);
    portoStat.SetMemoryGuarantee(stats.MemoryGuarantee);
    portoStat.SetMemoryGuaranteeTotal(stats.MemoryGuaranteeTotal);
    portoStat.SetAnonLimit(stats.AnonLimit);
    portoStat.SetAnonLimitTotal(stats.AnonLimitTotal);
    portoStat.SetMinorPageFaults(stats.MinorPageFaults);
    portoStat.SetMajorPageFaults(stats.MajorPageFaults);
    portoStat.SetOomKills(stats.OomKills);
    portoStat.SetOomKillsTotal(stats.OomKillsTotal);
    portoStat.SetDiskIoReadBytes(stats.DiskIoReadBytes);
    portoStat.SetDiskIoWriteBytes(stats.DiskIoWriteBytes);
    portoStat.SetDiskIoTimeMillis(stats.DiskIoTimeMillis);
    portoStat.SetTimeSeconds(stats.TimeSeconds);

    for (auto&& n: stats.Networks) {
        auto* portoNet = portoStat.AddIfs();
        portoNet->SetDev(n.first);
        portoNet->SetTxBytes(n.second.TxBytes);
        portoNet->SetRxBytes(n.second.RxBytes);
        portoNet->SetTxDrops(n.second.TxDrop);
        portoNet->SetRxDrops(n.second.RxDrop);
        portoNet->SetTxPackets(n.second.TxPackets);
        portoNet->SetRxPackets(n.second.RxPackets);
    }

    for (auto&& v: volumes) {
        auto* volume = portoStat.AddVolumes();
        volume->SetPath(v.Path);
        volume->SetFreeBytes(v.SpaceAvailableBytes);
        volume->SetUsedBytes(v.SpaceUsedBytes);
        volume->SetSizeBytes(v.SpaceLimitBytes);
    }

    return portoStat;
}

} // namespace

static constexpr auto MAPPINGS_LIMIT = 16u;

Porto GetPortoStats(const TVector<TString>& mappings) {
    Y_ENSURE(mappings.size() < MAPPINGS_LIMIT, "mappings size is " << mappings.size()
        << " but the limit is " << MAPPINGS_LIMIT);

    const TString rawRegex {mappings.size() > 0
        ? MakeRegexp(mappings)
        : ANY};

    re2::RE2 re{rawRegex};

    Y_ENSURE(re.ok(), "Unable to compile regex " << rawRegex << ": " << re.error());

    const auto portoClient = NSolomon::CreatePortoStatClient();
    const auto containers = portoClient->RunningContainers({});

    // number of capture groups is defined at runtime, so we allocate a pool for them
    std::array<re2::StringPiece, MAPPINGS_LIMIT> captures;
    std::array<re2::RE2::Arg, MAPPINGS_LIMIT> args;
    std::array<re2::RE2::Arg*, MAPPINGS_LIMIT> argPointers;

    for (auto i = 0u; i < MAPPINGS_LIMIT; ++i) {
        args[i] = &captures[i];
        argPointers[i] = &args[i];
    }

    Porto result;
    for (auto&& c: containers) {
        TString name;
        if (mappings.size() > 0) {
            if (!RE2::FullMatchN({c}, re, &argPointers.front(), re.NumberOfCapturingGroups())) {
                continue;
            }

            // first non-empty capture group is what we looking for
            for (auto i = 0; i < re.NumberOfCapturingGroups(); ++i) {
                if (!captures[i].empty()) {
                    name = captures[i].ToString();
                    break;
                }
            }
        } else {
            name = c;
        }

        auto&& stats = portoClient->Stats(c);
        result.AddContainers()->CopyFrom(StatsToProto(name, stats));
    }


    return result;
}

Porto GetPortoSelf(std::optional<TString> volumeNameRegex) {
    const auto portoClient = NSolomon::CreatePortoStatClient();

    Porto result;
    auto&& stats = portoClient->Stats("self");

    TVector<NSolomon::TVolumeStats> volumes;
    if (volumeNameRegex) {
        volumes = GetVolumeStats(*portoClient, "self", *volumeNameRegex);
    }

    result.AddContainers()->CopyFrom(StatsToProto("self", stats, volumes));

    return result;
}

Porto GetPortoSlot(const TString& portoSlotName) {
    const auto portoClient = NSolomon::CreatePortoStatClient();

    Porto result;
    auto&& stats = portoClient->Stats(portoSlotName);
    auto&& protoStat = StatsToProto("slot", stats);
    result.AddContainers()->Swap(&protoStat);

    return result;
}

} // namespace NSolomon
