#include "proc.h"

#include <util/stream/file.h>
#include <util/generic/vector.h>
#include <util/generic/map.h>
#include <util/generic/string.h>
#include <util/string/split.h>

//
// Created by nohttp on 14.04.16.
//

using namespace NProtoBuf;

namespace NSolomon {
namespace {

typedef struct CoreInfo {
    uint32_t PhysicalCpu;
    float Frequency;
} CoreInfo;

void CalculateFrequency(SysmonCpu *c, TVector<CoreInfo *> *cores) {
    TMap < uint32_t, TVector < CoreInfo * > * > byPhysicalCpu;
    typedef TMap<uint32_t, TVector < CoreInfo * >*>::iterator map_it;
    for (TVector<CoreInfo *>::iterator it = cores->begin(); it != cores->end(); ++it) {
        TVector < CoreInfo * > *collector;
        map_it search = byPhysicalCpu.find((*it)->PhysicalCpu);
        if (search == byPhysicalCpu.end()) {
            collector = new TVector<CoreInfo *>();
            byPhysicalCpu.insert(std::make_pair((*it)->PhysicalCpu, collector));
        } else {
            collector = search->second;
        }
        collector->push_back(*it);
    }

    for (map_it mit = byPhysicalCpu.begin(); mit != byPhysicalCpu.end(); mit++) {
        float min = 0;
        float max = 0;
        float sum = 0;
        uint32_t num = 0;
        for (TVector<CoreInfo *>::iterator it = mit->second->begin(); it != mit->second->end(); ++it) {
            float frequency = (*it)->Frequency;
            if (num == 0) {
                min = frequency;
                max = frequency;
            } else {
                min = Min(min, frequency);
                max = Max(max, frequency);
            }
            sum += frequency;
            num++;
        }

        if (num != 0) {
            SysmonPhysicalCpuFrequency *pcf = c->AddPhysicalCpuFrequency();

            pcf->SetPhysicalCpuId(mit->first);
            pcf->SetMin(min);
            pcf->SetMax(max);
            pcf->SetAvg(sum / num);
        } else {
            // What a terrible failure!
        }
    }
    for (map_it mit = byPhysicalCpu.begin(); mit != byPhysicalCpu.end(); mit++) {
        delete mit->second;
    }
}

TString tryRead(TString *input, const char *prefix) {
    const char *separator = ": ";
    if (input->StartsWith(prefix)) {
        int32_t delim = input->find(separator);
        if (delim >= 0) {
            return input->substr(delim + strlen(separator));
        }
    }
    return nullptr;
}

#define TRY_READ_CPUINFO_FIELD(line, target, field, type, header)\
    {\
        TString value = tryRead(line, header);\
        if (value != nullptr) {\
            target->field = FromString<type>(value);\
        }\
    }

void DoWork(SysmonCpu *c, TVector<CoreInfo *> *cores){
    CalculateFrequency(c, cores);
    // Insert other parsers here...
}

} // namespace

void ProcCpuinfo(Sysmon *r, const TFsPath& procDirectory) {
    SysmonCpu *c = r->MutableCpu();
    TVector < CoreInfo * > cores;
    try {
        TUnbufferedFileInput is(procDirectory / "cpuinfo");
        TString line;
        CoreInfo *core = new CoreInfo;
        while (is.ReadLine(line)) {
            if (line.empty()) {
                cores.push_back(core);
                core = new CoreInfo;
            } else {
                TRY_READ_CPUINFO_FIELD(&line, core, PhysicalCpu, ui32, "physical id");
                TRY_READ_CPUINFO_FIELD(&line, core, Frequency, float, "cpu MHz");
            }
        }
        // /proc/cpuinfo ends with an empty line, so this last core definitely has no data
        delete core;

        DoWork(c, &cores);

        while (!cores.empty()) {
            delete cores.back();
            cores.pop_back();
        }
    } catch (...) {
    }
}

} // namespace NSolomon
