#include "proc.h"

#include <util/stream/file.h>
#include <util/generic/vector.h>
#include <util/string/split.h>

using namespace NProtoBuf;

namespace NSolomon {
namespace {

// protobuf part can be generated with somenthing like
// for proto in `cat /proc/net/snmp|sed -e "s/:.*//g"|uniq`; do echo "message SysmonSnmp$proto {"; i=1; for attr in `cat /proc/net/snmp|grep "^$proto:"|head -n 1|sed -e "s/.*: //g"`; do echo "    optional uint64 $attr = $i [ (NMonitoring.Metric).Mode = SENSOR_DERIV ];"; i=`expr $i + 1`;done; echo -e "}\n"; done;
// And then fix fields which are not ui64 like /proc/net/snmp's Tcp::MaxConn
void ProcessProtocolStats(Message *r, TString &headersRaw, TString &dataRaw) {
    const Descriptor *desc = r->GetDescriptor();
    const Reflection *writer = r->GetReflection();

    TVector<TString> headers;
    Split(headersRaw, " ", headers);

    TVector<TString> data;
    Split(dataRaw, " ", data);

    if (data.size() == headers.size()) {
        for (ui64 i = 0; i < data.size(); i++) {
            const FieldDescriptor *field_desc = desc->FindFieldByName(headers[i]);
            if (field_desc != nullptr) {
                if (field_desc->type() == FieldDescriptor::TYPE_UINT64) {
                    writer->SetUInt64(r, field_desc, FromString<ui64>(data[i]));
                } else if (field_desc->type() == FieldDescriptor::TYPE_SINT64) {
                    writer->SetInt64(r, field_desc, FromString<i64>(data[i]));
                }
            }
        }
    }
}

void ProcessStats(Message *collector, TString &headers, TString &data) {
    const Descriptor *desc = collector->GetDescriptor();
    const Reflection *writer = collector->GetReflection();

    size_t prefix_len = headers.find_first_of(':');
    if (prefix_len > 0) {
        auto prefix = headers.substr(0, prefix_len);
        if (data.StartsWith(prefix + ": ")) {
            auto headersRaw = headers.substr(prefix.size() + 2);
            auto dataRaw = data.substr(prefix.size() + 2);
            const FieldDescriptor *field_desc = desc->FindFieldByName(prefix);
            if (field_desc != nullptr) {
                Message *protocol = writer->MutableMessage(collector, field_desc);
                ProcessProtocolStats(protocol, headersRaw, dataRaw);
            }
        }
    }
}

void ProcessStatsSimple(Message *collector, TString &data) {
    const Descriptor *desc = collector->GetDescriptor();
    const Reflection *writer = collector->GetReflection();

    TVector<TString> parts;
    Split(data, " \t", parts);

    const FieldDescriptor *field_desc = desc->FindFieldByName(parts[0]);
    if (field_desc != nullptr) {
        if (field_desc->type() == FieldDescriptor::TYPE_UINT64) {
            writer->SetUInt64(collector, field_desc, FromString<ui64>(parts[1]));
        }
    }

}

void ProcNetStatsFile(Message *collector, const TString &file) {
    try {
        TUnbufferedFileInput is(file);
        TString headers, data;
        while (is.ReadLine(headers) && is.ReadLine(data)) {
            ProcessStats(collector, headers, data);
        }
    } catch (...) {
    }
}

void ProcNetStatsFileSimple(Message *collector, const TString &file) {
    try {
        TUnbufferedFileInput is(file);
        TString data;
        while (is.ReadLine(data)) {
            ProcessStatsSimple(collector, data);
        }
    } catch (...) {
    }
}

void ProcNetSockstat(SysmonNet *net, const TString &file) {
    try {
        TFileInput is(file);
        auto firstLine = is.ReadLine();
        auto prefix = "sockets: used ";
        if (firstLine.StartsWith(prefix)) {
            auto valueStr = firstLine.substr(strlen(prefix));
            net->SetSocketsUsed(FromString<ui64>(valueStr));
        }

        TString line;
        while (is.ReadLine(line)) {
            TVector<TStringBuf> parts;
            Split(line, " ", parts);
            TStringBuf protocolName = parts[0];
            SysmonNetSockstatProtocol* protocol = net->AddProtocolSocketState();
            protocol->SetProtocol(TString(protocolName).substr(0, protocolName.length() - 1));

            const Reflection *writer = protocol->GetReflection();
            const Descriptor *desc = protocol->GetDescriptor();

            for (ui64 i = 1; i < parts.size(); i += 2) {
                TString name = TString(parts[i]);
                const FieldDescriptor *field_desc = desc->FindFieldByName(TString(parts[i]));
                if (field_desc != nullptr) {
                    writer->SetUInt64(protocol, field_desc, FromString<ui64>(parts[i+1]));
                }
            }
        }
    } catch (...) {
    }
}

void ProcNetDev(SysmonNet* net, const TString &file) {
//
// Inter-|   Receive                                                |  Transmit
//  face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
//    lo: 20363508956 8376364    0    0    0     0          0         0 20363508956 8376364    0    0    0     0       0          0
//    eth1:       0       0    0    0    0     0          0         0        0       0    0    0    0     0       0          0

    try {
        TFileInput is(file);
        if (!is.ReadLine().StartsWith("Inter-|")) {
            return;
        }
        if (!is.ReadLine().StartsWith(" face |")) {
            return;
        }

        TString s;
        while (is.ReadLine(s)) {
            TStringBuf name, line;

            if (!TStringBuf{s}.TrySplit(':', name, line)) {
                continue;
            }

            TVector<TStringBuf> numbers;
            StringSplitter(line).Split(' ').SkipEmpty().ParseInto(&numbers);
            if (numbers.size() != 16) {
                continue;
            }

            SysmonNetIf* ifProto = net->AddIfs();
            name.Skip(name.find_first_not_of(' '));
            TString nameStr{name};
            ifProto->SetDev(nameStr);
            ifProto->SetRxBytes(FromString(numbers[0]));
            ifProto->SetRxPackets(FromString(numbers[1]));
            ifProto->SetRxErrs(FromString(numbers[2]));
            ifProto->SetRxDrop(FromString(numbers[3]));
            ifProto->SetTxBytes(FromString(numbers[8]));
            ifProto->SetTxPackets(FromString(numbers[9]));
            ifProto->SetTxErrs(FromString(numbers[10]));
            ifProto->SetTxDrop(FromString(numbers[11]));
        }
    } catch (...) {
    }
}

} // namespace

void ProcNetBasic(Sysmon* sysmon, const TFsPath& procDirectory) {
    Y_ENSURE(sysmon != nullptr);
    ProcNetDev(sysmon->MutableNet(), procDirectory / "net/dev");
}

void ProcNetAdvanced(Sysmon* sysmon, const TFsPath& procDirectory) {
    Y_ENSURE(sysmon != nullptr);
    ProcNet(sysmon->MutableNet(), procDirectory);
}

void ProcNetNetstat(SysmonNet *net, const TString &file) {
    ProcNetStatsFile(net->MutableNetstat(), file);
}

void ProcNetSnmp(SysmonNet *net, const TString &file) {
    ProcNetStatsFile(net->MutableSnmp(), file);
}

void ProcNetSnmp6(SysmonNet *net, const TString &file) {
    ProcNetStatsFileSimple(net->MutableSnmp6(), file);
}

void ProcNet(SysmonNet* net, const TFsPath& procDirectory) {
    ProcNetDev(net, procDirectory / "net/dev");
    ProcNetNetstat(net, procDirectory / "net/netstat");
    ProcNetSnmp(net, procDirectory / "net/snmp");
    ProcNetSnmp6(net, procDirectory / "net/snmp6");
    ProcNetSockstat(net, procDirectory / "net/sockstat");
    ProcNetSockstat(net, procDirectory / "net/sockstat6");
}

} // namespace NSolomon
