#include <solomon/libs/cpp/sysmon/proc.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/platform.h>
#include <util/stream/file.h>

using namespace NSolomon;

TEST(TSysmonFillTest, ProcUptime) {
    Sysmon sysmon = FillSysmon();
    ASSERT_TRUE(sysmon.GetSystem().HasUpTime());
    ASSERT_TRUE(sysmon.GetSystem().HasUpTimeRaw());
}

TEST(TSysmonFillTest, ProcStat) {
    Sysmon sysmon = FillSysmon();
    if (!sysmon.GetSystem().HasUpTime()) {
        return;
    }
    ASSERT_TRUE(sysmon.GetSystem().HasNiceTime());
    ASSERT_TRUE(sysmon.GetSystemCpu().GetNode(0).HasNiceTime());
    ASSERT_TRUE(sysmon.GetSystem().GetUpTime() > 0);

    ASSERT_TRUE(sysmon.GetProc().HasProcsSinceBoot());
    ASSERT_TRUE(sysmon.GetProc().GetProcsSinceBoot() > 0);
}

TEST(TSysmonFillTest, ProcLoadavg) {
    Sysmon sysmon = FillSysmon();
    ASSERT_TRUE(sysmon.GetProc().HasLa());

    ASSERT_TRUE(sysmon.GetProc().HasLoadAverage1min());
    ASSERT_TRUE(sysmon.GetProc().HasLoadAverage5min());
    ASSERT_TRUE(sysmon.GetProc().HasLoadAverage15min());
    ASSERT_GT(sysmon.GetProc().GetLoadAverage1min(), 0);

    ASSERT_TRUE(sysmon.GetProc().HasThreads());
    ASSERT_GT(sysmon.GetProc().GetThreads(), 2u);
}

TEST(TSysmonFillTest, ProcMeminfo) {
    Sysmon sysmon = FillSysmon();
    ASSERT_TRUE(sysmon.GetMemory().HasMemTotal());
    ASSERT_TRUE(sysmon.GetMemory().HasMemFree());
    ASSERT_GT(sysmon.GetMemory().GetMemTotal(), (1u << 20));
    ASSERT_GT(sysmon.GetMemory().GetMemTotal(), sysmon.GetMemory().GetMemFree());
}

TEST(TSysmonFillTest, ProcVmstat) {
    Sysmon sysmon = FillSysmon();
    ASSERT_TRUE(sysmon.GetMemory().HasMajorPageFaults());
}

TEST(TSysmonFillTest, ProcNetDev) {
    Sysmon sysmon = FillSysmon();
    for (long i = 0; i < sysmon.GetNet().GetIfs().size(); ++i) {
        const SysmonNetIf& ifProto = sysmon.GetNet().GetIfs(i);
        if (ifProto.GetDev() == "lo") {
            ASSERT_TRUE(ifProto.HasRxBytes());
            ASSERT_TRUE(ifProto.HasTxPackets());
            return;
        }
    }
    FAIL() << "lo not found";
}

TEST(TSysmonFillTest, ProcNetSockstat) {
    Sysmon sysmon = FillSysmon();
    ASSERT_GT(sysmon.GetNet().GetSocketsUsed(), 0u);
}

TEST(TSysmonFillTest, ProcNetExt) {
    Sysmon sysmon = FillSysmon();
    ASSERT_GT(sysmon.GetNet().GetNetstat().GetTcpExt().GetTW(), 0u);
    ASSERT_GT(sysmon.GetNet().GetNetstat().GetIpExt().GetInOctets(), 0u);
}

TEST(TSysmonFillTest, Statvfs) {
    Sysmon sysmon = FillSysmon();
    for (long i = 0; i < sysmon.GetFilesystem().GetPoints().size(); ++i) {
        const SysmonMountpoint& mountpointProto = sysmon.GetFilesystem().GetPoints(i);
        if (mountpointProto.GetDev() == "/") {
            ASSERT_TRUE(mountpointProto.HasSizeB());
            ASSERT_TRUE(mountpointProto.HasUsedB());
            ASSERT_TRUE(mountpointProto.HasFreeB());
        }
    }
}

TEST(TSysmonFillTest, Cpuinfo) {
    Sysmon sysmon = FillSysmon();
//    ASSERT_FALSE(sysmon.GetCpu().GetPhysicalCpuFrequency().empty());
}

TEST(TSysmonFillTest, ProcDiskstats) {
    Sysmon sysmon = FillSysmon();

    for (auto disk: sysmon.GetIo().GetDisks()) {
        ASSERT_TRUE(disk.HasDev());

        ASSERT_TRUE(disk.HasReads());
        ASSERT_TRUE(disk.HasReadBytes());
        ASSERT_TRUE(disk.HasReadWaitMillisec());

        ASSERT_TRUE(disk.HasWrites());
        ASSERT_TRUE(disk.HasWriteBytes());
        ASSERT_TRUE(disk.HasWriteWaitMillisec());

        ASSERT_TRUE(disk.HasIOsInProgress());
        ASSERT_TRUE(disk.HasIOMillisec());
        ASSERT_TRUE(disk.HasTimeInQueueMillisec());
        ASSERT_TRUE(disk.HasIOServiceTime());

        ASSERT_TRUE(disk.HasReadWaitMillisecAvg());
        ASSERT_TRUE(disk.HasWriteWaitMillisecAvg());
    }
}

TEST(TSysmonFillTest, ProcDiskstatsNvmeInfo) {
    auto diskstats = TFileOutput("./diskstats");
    diskstats << R"(0       0 nvme1c1n1 1191515097 252212808 187575087706 467523567 991708012 712845962 129972916376 128625135 0 528389683 674288198 283044406 0 452104584824 78139495 0 0
0       0 nvme11c11n1 103 0 4776 6 0 0 0 0 0 15 6 0 0 0 0 0 0
259       1 nvme1n1 9931933 0 614913012 3311796 737250439 0 66505968316 2106629380 0 28705012 2408237992
259       1 nvme11n1 9931934 0 614913012 3311796 737250439 0 66505968316 2106629380 0 28705012 2408237992)";
    diskstats.Flush();
    THashMap<TString, size_t> expected = {
            {"nvme1c1n1", 1191515097},
            {"nvme11c11n1", 103},
            {"nvme1n1", 9931933},
            {"nvme11n1", 9931934},
    };

    NSolomon::Sysmon sysmon{};
    ProcDiskstats(&sysmon, TFsPath("."));
    for (auto disk: sysmon.GetIo().GetDisks()) {
        ASSERT_TRUE(disk.HasDev());
        ASSERT_TRUE(expected.contains(disk.dev()));
        ASSERT_EQ(expected[disk.dev()], disk.reads());
    }
    ASSERT_EQ(expected.size(), static_cast<size_t>(sysmon.GetIo().disks_size()));
}
