#pragma once

#include <solomon/libs/cpp/threading/pool/pool.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon {
struct TThreadPoolMetrics: public IThreadPoolStatusListener {
public:
    TThreadPoolMetrics(TStringBuf name, NMonitoring::TMetricRegistry& registry) {
        MaxSize_ = registry.IntGauge({{"sensor", "threadPool.maxSize"}, {"id", name}});
        RunningThreads_ = registry.IntGauge({{"sensor", "threadPool.runningThreads"}, {"id", name}});
        QueuedTasks_ = registry.IntGauge({{"sensor", "threadPool.queuedTasks"}, {"id", name}});
        SubmittedTasks_ = registry.Rate({{"sensor", "threadPool.submittedTasks"}, {"id", name}});
        CompletedTasks_ = registry.Rate({{"sensor", "threadPool.completedTasks"}, {"id", name}});
        WaitTimeMs_ = registry.HistogramRate({{"sensor", "threadPool.waitTimeMs"}, {"id", name}},
            NMonitoring::ExponentialHistogram(16, 2, 1));
        ExecTimeMs_ = registry.HistogramRate({{"sensor", "threadPool.execTimeMs"}, {"id", name}},
            NMonitoring::ExponentialHistogram(16, 2, 1));
    }

private:
    void OnStart(size_t threadCount, size_t) noexcept override {
        MaxSize_->Set(threadCount);
    }

    void OnTaskScheduled() noexcept override {
        QueuedTasks_->Inc();
        SubmittedTasks_->Inc();
    }

    void OnTaskRejected() noexcept override {
    }

    void OnTaskBeforeExecution(TDuration waitTime) noexcept override {
        RunningThreads_->Inc();
        WaitTimeMs_->Record(waitTime.MilliSeconds());
    }

    void OnTaskCompleted(TDuration executionTime) noexcept override {
        RunningThreads_->Dec();
        CompletedTasks_->Inc();
        QueuedTasks_->Dec();
        ExecTimeMs_->Record(executionTime.MilliSeconds());
    }

    void OnStop() noexcept override {
    };

private:
    NMonitoring::TIntGauge* MaxSize_;
    NMonitoring::TIntGauge* RunningThreads_;
    NMonitoring::TIntGauge* QueuedTasks_;
    NMonitoring::TRate* SubmittedTasks_;
    NMonitoring::TRate* CompletedTasks_;
    NMonitoring::THistogram* WaitTimeMs_;
    NMonitoring::THistogram* ExecTimeMs_;
};
} // namespace NSolomon
