#include "pool.h"

#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <chrono>

namespace NSolomon {
namespace {

class TObjectInQueueWrapper: public IObjectInQueue {
public:
    explicit TObjectInQueueWrapper(
            IObjectInQueue* obj,
            IThreadPoolStatusListenerPtr statusListener = nullptr)
        : Obj_(obj)
        , StatusListener_(std::move(statusListener))
    {}

    ~TObjectInQueueWrapper() override {
        if (StatusListener_) {
            StatusListener_->OnTaskCompleted(Timer_.Step());
        }
    }

    // Expected to be called only once and only by one thread at most, since it's a queue task
    void Process(void* threadSpecificResource) override {
        THolder<TObjectInQueueWrapper> this_(this);

        if (StatusListener_) {
            StatusListener_->OnTaskBeforeExecution(Timer_.Step());
        }

        Obj_->Process(threadSpecificResource);
    }

private:
    IObjectInQueue* Obj_;
    IThreadPoolStatusListenerPtr StatusListener_;

    // TODO: optimization -- compute an actual user time for OnTaskCompleted
    TSteadyTimer Timer_;
};

class TThreadPoolProxy: public IThreadPool {
public:
    explicit TThreadPoolProxy(
            IThreadPool* pool,
            IThreadPoolStatusListenerPtr statusListener = nullptr)
        : Pool_{pool}
        , StatusListener_{std::move(statusListener)}
    {}

    explicit TThreadPoolProxy(
            std::unique_ptr<IThreadPool> pool,
            IThreadPoolStatusListenerPtr statusListener = nullptr)
        : Pool_{pool.get()}
        , OwnedPool_{std::move(pool)}
        , StatusListener_{std::move(statusListener)}
    {}

private:
    bool Add(IObjectInQueue* obj) override {
        auto wrappedObjPtr = std::make_unique<TObjectInQueueWrapper>(obj, StatusListener_);
        bool added = Pool_->Add(wrappedObjPtr.get());

        if (added) {
            // an obj will be deleted by itself, after TObjectInQueueWrapper::Process() completion
            Y_UNUSED(wrappedObjPtr.release());

        }

        if (StatusListener_) {
            if (added) {
                StatusListener_->OnTaskScheduled();
            } else {
                StatusListener_->OnTaskRejected();
            }
        }

        return added;
    }

    void Start(size_t threadCount, size_t queueSizeLimit) override {
        Pool_->Start(threadCount, queueSizeLimit);

        if (StatusListener_) {
            StatusListener_->OnStart(threadCount, queueSizeLimit);
        }
    }

    void Stop() noexcept override {
        Pool_->Stop();

        if (StatusListener_) {
            StatusListener_->OnStop();
        }
    }

    size_t Size() const noexcept override {
        return Pool_->Size();
    }

    void* CreateThreadSpecificResource() override {
        return Pool_->CreateThreadSpecificResource();
    }

    void DestroyThreadSpecificResource(void* resource) override {
        Pool_->DestroyThreadSpecificResource(resource);
    }

private:
    IThreadPool* Pool_;
    std::unique_ptr<IThreadPool> OwnedPool_;
    IThreadPoolStatusListenerPtr StatusListener_;
};

} // namespace

std::shared_ptr<IThreadPool> CreateThreadPool(
        size_t threadCount,
        size_t queueSizeLimit,
        const TThreadPool::TParams& params,
        IThreadPoolStatusListenerPtr statusListener)
{
    std::shared_ptr<IThreadPool> pool =
        std::make_shared<TThreadPoolProxy>(std::make_unique<TThreadPool>(params), std::move(statusListener));
    pool->Start(threadCount, queueSizeLimit);
    return pool;
}

std::shared_ptr<IThreadPool> CreateThreadPoolProxy(
        IThreadPool* pool,
        IThreadPoolStatusListenerPtr statusListener)
{
    return std::make_shared<TThreadPoolProxy>(pool, std::move(statusListener));
}

std::unique_ptr<IThreadPool> CreateThreadPoolProxyUnique(
    std::unique_ptr<IThreadPool> pool,
    IThreadPoolStatusListenerPtr statusListener)
{
    return std::make_unique<TThreadPoolProxy>(std::move(pool), std::move(statusListener));
}

} // namespace NSolomon
