#include <solomon/libs/cpp/threading/pool/pool.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/thread.h>

using namespace NSolomon;

enum class ETaskState {
    NotInitialized = 0,
    Scheduled,
    Running,
    Completed,
};

class TStatusListener: public NSolomon::IThreadPoolStatusListener {
private:
    void OnStart(size_t threadCount, size_t queueSizeLimit) noexcept override {
        IsRunning = true;
        ThreadCount = threadCount;
        QueueSizeLimit = queueSizeLimit;
    }

    void OnTaskScheduled() noexcept override {
        ++TasksScheduledCnt;
        TaskState = ETaskState::Scheduled;
    }

    void OnTaskRejected() noexcept override {
        // there's no good way to test it
    }

    void OnTaskBeforeExecution(TDuration) noexcept override {
        TaskState = ETaskState::Running;
        ++TasksBeforeExecutionCnt;
    }

    void OnTaskCompleted(TDuration execRealTime) noexcept override {
        ++TasksCompletedCnt;
        ExecRealTime = execRealTime;
        TaskState = ETaskState::Completed;
    }

    void OnStop() noexcept override {
        IsRunning = false;
    }

public:
    std::atomic<bool> IsRunning{false};
    std::atomic<size_t> ThreadCount{0};
    std::atomic<size_t> QueueSizeLimit{std::numeric_limits<size_t>::max()};

    std::atomic<ui32> TasksCompletedCnt{0};
    std::atomic<ui32> TasksScheduledCnt{0};
    std::atomic<ui32> TasksBeforeExecutionCnt{0};

    std::atomic<TDuration> ExecRealTime{TDuration::Zero()};
    std::atomic<ETaskState> TaskState{ETaskState::NotInitialized};
};

TEST(ThreadPoolTest, Proxy) {
    auto sl = std::make_shared<TStatusListener>();

    TThreadPool p(TThreadPool::TParams().SetBlocking(false).SetCatching(true));

    auto proxy = NSolomon::CreateThreadPoolProxy(&p, sl);

    ASSERT_EQ(sl->IsRunning.load(), false);
    ASSERT_EQ(sl->ThreadCount.load(), 0ul);
    ASSERT_EQ(sl->QueueSizeLimit.load(), std::numeric_limits<size_t>::max());
    ASSERT_EQ(sl->TasksCompletedCnt.load(), 0ul);
    ASSERT_EQ(sl->TasksBeforeExecutionCnt.load(), 0ul);

    proxy->Start(2, 0);
    ASSERT_EQ(sl->IsRunning.load(), true);
    ASSERT_EQ(sl->ThreadCount.load(), 2ul);
    ASSERT_EQ(sl->QueueSizeLimit.load(), 0ul);
    ASSERT_EQ(sl->TaskState.load(), ETaskState::NotInitialized);

    ASSERT_TRUE(proxy->AddFunc([]() {
        Sleep(TDuration::MilliSeconds(1'200));
    }));

    ASSERT_EQ(sl->TasksScheduledCnt.load(), 1ul);
    ASSERT_EQ(sl->TasksCompletedCnt.load(), 0ul);
    auto taskState = sl->TaskState.load();
    ASSERT_TRUE(taskState == ETaskState::Scheduled || taskState == ETaskState::Running);

    Sleep(TDuration::MilliSeconds(700));
    ASSERT_EQ(sl->TaskState.load(), ETaskState::Running);

    Sleep(TDuration::MilliSeconds(800));
    ASSERT_EQ(sl->TasksScheduledCnt.load(), 1ul);
    ASSERT_EQ(sl->TasksCompletedCnt.load(), 1ul);
    ASSERT_EQ(sl->TasksBeforeExecutionCnt.load(), 1ul);
    ASSERT_TRUE(sl->ExecRealTime.load().MilliSeconds() > 1'000);
    ASSERT_EQ(sl->TaskState.load(), ETaskState::Completed);

    proxy->Stop();
    ASSERT_EQ(sl->IsRunning.load(), false);
}

TEST(ThreadPoolTest, ProxyWithOwning) {
    std::atomic_bool isDeleted{false};

    class TThreadPoolWithCustomDtor: public TThreadPool {
    public:
        explicit TThreadPoolWithCustomDtor(std::atomic_bool& isDeleted)
            : IsDeleted_{isDeleted}
        {
        }

        ~TThreadPoolWithCustomDtor() override {
            IsDeleted_ = true;
        }

    private:
        std::atomic_bool& IsDeleted_;
    };

    auto poolPtr = std::make_unique<TThreadPoolWithCustomDtor>(isDeleted);
    ASSERT_EQ(isDeleted.load(), false);

    CreateThreadPoolProxy(poolPtr.get());
    ASSERT_EQ(isDeleted.load(), false);

    CreateThreadPoolProxyUnique(std::move(poolPtr));
    ASSERT_TRUE(isDeleted.load()) << "proxy owning an impl didn't destroy an impl";
}
