#include "timer_wheel.h"

#include <util/system/yassert.h>

namespace NSolomon::NTimer {
namespace {

// convert TInstant or TDuration to a number of ticks
template <typename T>
constexpr ui64 ToTicks(T instantOrDuration, TDuration tickDuration) {
    return instantOrDuration.GetValue() / tickDuration.GetValue();
}

// convert number of ticks to a TInstant or TDuration
template <typename T>
constexpr T FromTicks(ui64 ticks, TDuration tickDuration) {
    return T::FromValue(ticks * tickDuration.GetValue());
}

} // namespace

TTimerWheel::TTimerWheel(TInstant now, TDuration tickDuration)
    : TickDuration_{tickDuration}
    , TicksPending_{0}
{
    Y_ENSURE(tickDuration > TDuration::Zero(), "tick duration must be greater than 0");

    auto nowTick = ToTicks(now, TickDuration_);
    for (int i = 0; i < NUM_LEVELS; ++i) {
        NowTick_[i] = nowTick >> (WIDTH_BITS * i);
    }
}

void TTimerWheel::Advance(TDuration delta) {
    Y_ENSURE(delta >= TickDuration_, "delta (" << delta << ") is less than a tick duration (" << TickDuration_ << ')');
    ui64 deltaTicks = ToTicks(delta, TickDuration_);
    AdvanceInLevel(deltaTicks, 0);
}

void TTimerWheel::ScheduleAfter(ITimerEvent* event, TDuration delay) {
    Y_ENSURE(delay >= TickDuration_, "delay (" << delay << ") is less than a tick duration (" << TickDuration_ << ')');
    auto delayTicks = ToTicks(delay, TickDuration_);
    event->ScheduledAt_ = FromTicks<TInstant>(NowTick_[0] + delayTicks, TickDuration_);
    event->Relink(FindSlot(delayTicks));
}

void TTimerWheel::ScheduleAt(ITimerEvent* event, TInstant time) {
    auto now = FromTicks<TInstant>(NowTick_[0] + 1, TickDuration_);
    Y_ENSURE(time >= now, "time (" << time << ") must be greater than now of the next tick (" << now << ")");

    auto timeTicks = ToTicks(time, TickDuration_);
    event->ScheduledAt_ = FromTicks<TInstant>(timeTicks, TickDuration_);
    event->Relink(FindSlot(timeTicks - NowTick_[0]));
}

void TTimerWheel::AdvanceInLevel(ui64 ticks, int level) {
    if (TicksPending_ == 0) {
        // Zero ticks are only ok when in the middle of a partially processed tick.
        Y_VERIFY_DEBUG(ticks > 0);
    } else {
        if (level == 0) {
            TicksPending_ += ticks;
        }

        auto nowTick = NowTick_[level];
        ProcessCurrentSlot(nowTick, level);

        if (level > 0) {
            return;
        }

        ticks = (TicksPending_ - 1);
        TicksPending_ = 0;
    }

    while (ticks--) {
        auto nowTick = ++NowTick_[level];
        ProcessCurrentSlot(nowTick, level);
    }
}

void TTimerWheel::ProcessCurrentSlot(ui64 nowTick, int level) {
    size_t slotIndex = nowTick & MASK;
    if (slotIndex == 0 && level < MAX_LEVEL) {
        AdvanceInLevel(1, level + 1);
    }

    auto* slot = &Slots_[level][slotIndex];
    if (level == 0) {
        while (slot->Events) {
            auto* event = slot->Pop();
            event->Derived()->Execute();
        }
    } else {
        while (slot->Events) {
            auto* item = slot->Pop();
            ITimerEvent* event = item->Derived();

            Y_VERIFY_DEBUG((NowTick_[0] & MASK) == 0);

            auto scheduledAtTick = ToTicks(event->ScheduledAt_, TickDuration_);
            if (NowTick_[0] >= scheduledAtTick) {
                event->Execute();
            } else {
                // promote event to a slot in first level
                event->Relink(FindSlot(scheduledAtTick - NowTick_[0]));
            }
        }
    }
}

TDuration TTimerWheel::DelayToNextEvent(TDuration max) {
    if (TicksPending_) {
        return TDuration::Zero();
    }

    TInstant now = FromTicks<TInstant>(NowTick_[0], TickDuration_);
    return DelayToNextEventInLevel(max, now, 0);
}

TDuration TTimerWheel::DelayToNextEventInLevel(TDuration max, TInstant now, int level) {
    TDuration min = max;

    for (int i = 0; i < NUM_SLOTS; ++i) {
        auto slotIndex = (NowTick_[level] + 1 + i) & MASK;
        if (slotIndex == 0 && level < MAX_LEVEL) {
            if (level > 0 || !Slots_[level][slotIndex].Events) {
                auto upSlotIndex = (NowTick_[level + 1] + 1) & MASK;
                const auto& slot = Slots_[level + 1][upSlotIndex];
                for (auto* event = slot.Events; event; event = event->Next_) {
                    min = std::min(min, event->Derived()->ScheduledAt() - now);
                }
            }
        }

        const auto& slot = Slots_[level][slotIndex];
        auto* event = slot.Events;

        // in first level we check only first slot
        if (level == 0 && event) {
            return std::min(min, event->Derived()->ScheduledAt() - now);
        }

        // on the rest levels we check all events
        if (event) {
            while (event) {
                min = std::min(min, event->Derived()->ScheduledAt() - now);
                event = event->Next_;
            }
            return min;
        }
    }

    // if nothing found on this level, try the next one (unless the level can't
    // possibly contain an event scheduled earlier than "max").
    if (level < MAX_LEVEL && (max.GetValue() >> (WIDTH_BITS * level + 1)) > 0) {
        return DelayToNextEventInLevel(max, now, level + 1);
    }

    return max;
}

TWheelSlot<ITimerEvent>* TTimerWheel::FindSlot(ui64 deltaTicks) {
    int level = 0;
    while (deltaTicks >= NUM_SLOTS) {
        deltaTicks = (deltaTicks + (NowTick_[level] & MASK)) >> WIDTH_BITS;
        ++level;
    }

    size_t slotIndex = (NowTick_[level] + deltaTicks) & MASK;
    return &Slots_[level][slotIndex];
}

} // namespace NSolomon::NTimer
