#pragma once

#include <util/datetime/base.h>
#include <util/generic/noncopyable.h>

namespace NSolomon::NTimer {

class ITimerEvent;
class TTimerWheel;

/**
 * TTimerWheel maintains a data structure which is effectively several levels of a ring buffers.
 * Each ring buffer contains several slots. Each slot corresponds to a specific timer tick, and
 * contains the head of a doubly-linked list. The linked list contains the events that should
 * happen on that tick.
 */
template <typename T>
struct TWheelSlot: private TNonCopyable {
    class TItem: private TNonCopyable {
        friend TTimerWheel;
        friend TWheelSlot;
    protected:
        // Current slot when event is registered.
        TWheelSlot* Slot_{nullptr};

        // Intrusive double-linked list of events in the slot.
        TItem* Next_{nullptr};
        TItem* Prev_{nullptr};

        T* Derived() {
            return static_cast<T*>(this);
        }

        const T* Derived() const {
            return static_cast<const T*>(this);
        }
    };

    TItem* Events{nullptr};

    TItem* Pop() {
        auto* event = Events;
        Events = event->Next_;
        if (Events) {
            Events->Prev_ = nullptr;
        }
        event->Next_ = nullptr;
        event->Slot_ = nullptr;
        return event;
    }

    void Push(TItem* event) {
        event->Next_ = Events;
        if (Events) {
            Events->Prev_ = event;
        }
        Events = event;
        event->Slot_ = this;
    }

    void Remove(TItem* event) {
        auto* prev = event->Prev_;
        auto* next = event->Next_;
        if (next) {
            next->Prev_ = prev;
            event->Next_ = nullptr;
        }
        if (prev) {
            prev->Next_ = next;
            event->Prev_ = nullptr;
        } else {
            Events = next;
        }
        event->Slot_ = nullptr;
    }
};

/**
 * A Timer optimized for approximated I/O timeout scheduling.
 * The main scenario this timer is optimizer for is fast schedule/cancel events, because
 * wast majority of them will never happen (pessimistic timeouts).
 *
 * You can increase or decrease the accuracy of the execution timing by specifying smaller
 * or larger tick duration in the constructor. In most network applications, I/O timeout
 * does not need to be accurate. Therefore, the default tick duration is 100 milliseconds
 * and you will not need to try different configurations in most cases.
 *
 * This implementation does not create a thread and does not automatically advance time pointer.
 * It allows you to use this timer with your own event loop.
 *
 * Right now this implementation is not thread-safe, because we plan to use it inside Actor,
 * meaning single-threaded execution. But thread synchronization can be easily added inside
 * wheel slots.
 *
 * Links:
 *   - http://www.cs.columbia.edu/~nahum/w6998/papers/sosp87-timing-wheels.pdf
 *   - https://blog.acolyer.org/2015/11/23/hashed-and-hierarchical-timing-wheels/
 *   - https://www.snellman.net/blog/archive/2016-07-27-ratas-hierarchical-timer-wheel/
 */
class TTimerWheel: private TNonCopyable {
    static constexpr int WIDTH_BITS = 8;
    static constexpr int NUM_LEVELS = (64 + WIDTH_BITS - 1) / WIDTH_BITS;
    static constexpr int MAX_LEVEL = NUM_LEVELS - 1;
    static constexpr int NUM_SLOTS = 1 << WIDTH_BITS;
    static constexpr int MASK = (NUM_SLOTS - 1);

public:
    /**
     * Create a new timer with provided current time and tick duration.
     *
     * @param now           current time, which will be rounded by tick duration
     * @param tickDuration  the duration between ticks
     */
    explicit TTimerWheel(TInstant now = TInstant::Zero(), TDuration tickDuration = TDuration::MilliSeconds(100));

    /**
     * Advance the timer by the specified amount of time, and execute
     * any events scheduled for execution at or before that time. This
     * method should not be called from an event Execute() method.
     *
     * @param delta amount of time to move on (must be > 0)
     */
    void Advance(TDuration delta);

    /**
     * Schedule the event for one-time execution after the specified delay.
     * The actual time will be adjusted by timer's tick duration.
     *
     * @param event  pointer to an event which must be scheduled, timer does not take
     *               ownership of an event object
     * @param delay  delay before an event will be executed (must be > 0).
     */
    void ScheduleAfter(ITimerEvent* event, TDuration delay);

    /**
     * Schedule the event for one-time execution at the specific time in the future some.
     * The actual time will be adjusted by timer's tick duration.
     *
     * @param event  pointer to an event which must be scheduled, timer does not take
     *               ownership of an event object
     * @param time   point of time in the future when event will be executed (mut be > timer.Now())
     */
    void ScheduleAt(ITimerEvent* event, TInstant time);

    /**
     * @return current time adjusted by timer's tick duration.
     */
    TInstant Now() const {
        return TInstant::FromValue(NowTick_[0] * TickDuration_.GetValue());
    }

    /**
     * @return configured tick duration.
     */
    TDuration TickDuration() const {
        return TickDuration_;
    }

    /**
     * Find the duration remaining until the next event will get executed. If the
     * max parameter is passed, that will be the maximum value that will be returned.
     * Also it will be returned if no events have been scheduled.
     *
     * @param max    maximum delay
     * @param level
     * @return
     */
    TDuration DelayToNextEvent(TDuration max = TDuration::Max());

private:
    void AdvanceInLevel(ui64 ticks, int level);
    void ProcessCurrentSlot(ui64 nowTick, int level);
    TDuration DelayToNextEventInLevel(TDuration max, TInstant now, int level);

    TWheelSlot<ITimerEvent>* FindSlot(ui64 deltaTicks);

private:
    const TDuration TickDuration_;
    ui64 TicksPending_;
    ui64 NowTick_[NUM_LEVELS];
    TWheelSlot<ITimerEvent> Slots_[NUM_LEVELS][NUM_SLOTS];
};

/**
 * Base class for all scheduling events.
 */
class ITimerEvent: public TWheelSlot<ITimerEvent>::TItem {
    friend TTimerWheel;
public:
    virtual ~ITimerEvent() {
        Cancel();
    }

    /**
     * Unschedule this event. It's safe to cancel an event that was never scheduled.
     */
    void Cancel() {
        if (IsScheduled()) {
            Relink(nullptr);
            ScheduledAt_ = TInstant::Zero();
        }
    }

    /**
     * @return true iff the event is currently scheduled for execution.
     */
    bool IsScheduled() const {
        return Slot_ != nullptr;
    }

    /**
     * @return the point in time when this event is scheduled to be executed.
     */
    TInstant ScheduledAt() const {
        return ScheduledAt_;
    }

    /**
     * Implement in subclasses. Executes the event callback.
     */
    virtual void Execute() = 0;

private:
    /**
     * Move the event to another slot.
     */
    void Relink(TWheelSlot<ITimerEvent>* slot) {
        if (Slot_) {
            Slot_->Remove(this);
        }
        if (slot) {
            slot->Push(this);
        }
    }

private:
    TInstant ScheduledAt_;
};

/**
 * An event that will invoke given functor on event execution.
 */
template <typename TFunc>
class TFuncEvent: public ITimerEvent {
public:
    explicit TFuncEvent(TFunc&& func)
        : Func_{std::forward<TFunc>(func)} {
    }

    void Execute() override {
        Func_();
    }

private:
    TFunc Func_;
};

/**
 * An event that will invoke member function on event execution.
 */
template <typename T, void(T::*MemFunc)() >
class TMemFuncEvent: public ITimerEvent {
public:
    explicit TMemFuncEvent(T* obj)
        : Obj_(obj)
    {
    }

    void Execute() override {
        (Obj_->*MemFunc)();
    }

private:
    T* Obj_;
};

} // namespace NSolomon::NTimer
