#include <solomon/libs/cpp/threading/timer/timer_wheel.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>
#include <util/datetime/cputimer.h>

#include <memory>

using namespace NSolomon::NTimer;

class TCounter: public ITimerEvent {
public:
    void Execute() override {
        Value_++;
    }

    size_t Value() const {
        return Value_;
    }

private:
    size_t Value_{0};
};

TEST(TTimerWheelTest, DefaultContructor) {
    TTimerWheel timer;
    ASSERT_EQ(timer.Now(), TInstant::Zero());
    ASSERT_EQ(timer.TickDuration(), TDuration::MilliSeconds(100));
}

TEST(TTimerWheelTest, NowIsRounededByTickDuration) {
    auto now = TInstant::Now();
    auto tick = TDuration::MilliSeconds(50);

    TTimerWheel timer{now, tick};
    ASSERT_EQ(timer.TickDuration(), tick);

    auto nowRounded = TInstant::FromValue(now.GetValue() - (now.GetValue() % tick.GetValue()));
    ASSERT_EQ(timer.Now(), nowRounded);
    ASSERT_LE(now - nowRounded, tick);
}

TEST(TTimerWheelTest, ScheduleAfter) {
    TTimerWheel timer;
    TCounter c1;
    TCounter c2;

    timer.ScheduleAfter(&c1, TDuration::Seconds(1));

    ASSERT_TRUE(c1.IsScheduled());
    ASSERT_EQ(c1.ScheduledAt(), timer.Now() + TDuration::MilliSeconds(1000));
    ASSERT_EQ(c1.Value(), 0u);

    timer.ScheduleAfter(&c2, TDuration::MilliSeconds(1300));
    ASSERT_TRUE(c2.IsScheduled());
    ASSERT_EQ(c2.ScheduledAt(), timer.Now() + TDuration::MilliSeconds(1300));
    ASSERT_EQ(c2.Value(), 0u);

    // (1) non of event are executed
    timer.Advance(TDuration::MilliSeconds(500));

    ASSERT_TRUE(c1.IsScheduled());
    ASSERT_EQ(c1.Value(), 0u);

    ASSERT_TRUE(c2.IsScheduled());
    ASSERT_EQ(c2.Value(), 0u);

    // (2) first event is executed, second - not
    timer.Advance(TDuration::MilliSeconds(500));

    ASSERT_FALSE(c1.IsScheduled());
    ASSERT_EQ(c1.Value(), 1u);

    ASSERT_TRUE(c2.IsScheduled());
    ASSERT_EQ(c2.Value(), 0u);

    // (3) now second event is executed
    timer.Advance(TDuration::MilliSeconds(500));

    ASSERT_FALSE(c2.IsScheduled());
    ASSERT_EQ(c2.Value(), 1u);
}

TEST(TTimerWheelTest, ScheduleAt) {
    TTimerWheel timer;
    TCounter c;

    auto at = TDuration::Seconds(10).ToDeadLine(timer.Now());
    timer.ScheduleAt(&c, at);

    ASSERT_TRUE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), at);
    ASSERT_EQ(c.Value(), 0u);

    timer.Advance(TDuration::Seconds(5));
    ASSERT_TRUE(c.IsScheduled());
    ASSERT_EQ(c.Value(), 0u);

    timer.Advance(TDuration::Seconds(5));
    ASSERT_FALSE(c.IsScheduled());
    ASSERT_EQ(c.Value(), 1u);
}

TEST(TTimerWheelTest, CancelNotScheduled) {
    TCounter c;
    ASSERT_FALSE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Zero());

    c.Cancel();
    ASSERT_FALSE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Zero());
}

TEST(TTimerWheelTest, CancelScheduled) {
    TCounter c;
    ASSERT_FALSE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Zero());

    TTimerWheel timer;
    timer.ScheduleAt(&c, TInstant::Seconds(20));
    ASSERT_TRUE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Seconds(20));

    c.Cancel();
    ASSERT_FALSE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Zero());
}

TEST(TTimerWheelTest, Reschedule) {
    TTimerWheel timer;
    TCounter c;

    timer.ScheduleAt(&c, TInstant::Seconds(20));
    ASSERT_TRUE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Seconds(20));

    // reschedule sooner
    timer.ScheduleAt(&c, TInstant::Seconds(10));
    ASSERT_TRUE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Seconds(10));

    // reschedule later
    timer.ScheduleAt(&c, TInstant::Seconds(30));
    ASSERT_TRUE(c.IsScheduled());
    ASSERT_EQ(c.ScheduledAt(), TInstant::Seconds(30));
}


TEST(TTimerWheelTest, DelayToNextEvent) {
    TTimerWheel timer;
    TCounter c1;
    TCounter c2;

    timer.ScheduleAt(&c1, TInstant::Seconds(1));
    timer.ScheduleAt(&c2, TInstant::Seconds(10));
    ASSERT_EQ(timer.DelayToNextEvent(), TDuration::Seconds(1));

    timer.Advance(TDuration::MilliSeconds(300));
    ASSERT_EQ(timer.DelayToNextEvent(), TDuration::MilliSeconds(700));

    timer.Advance(TDuration::MilliSeconds(700));
    ASSERT_EQ(timer.DelayToNextEvent(), TDuration::Seconds(9));

    c2.Cancel();
    ASSERT_EQ(timer.DelayToNextEvent(), TDuration::Max());
}

TEST(TTimerWheelTest, FuncExample) {
    TTimerWheel timer;
    bool executed = false;

    TFuncEvent event{[&]() {
        executed = true;
    }};

    timer.ScheduleAfter(&event, TDuration::Hours(1));
    timer.Advance(TDuration::Hours(1) - TDuration::MilliSeconds(1));
    ASSERT_FALSE(executed);

    timer.Advance(TDuration::MilliSeconds(100));
    ASSERT_TRUE(executed);
}

TEST(TTimerWheelTest, MemFuncExample) {
    // do not complicate structure interface, all details of event are hidden
    // do not additionally allocate event, it is stored inplace
    struct TMessage {
        bool IsSent() const {
            return Sent_;
        }

        void SendAfter(TTimerWheel* timer, TDuration delay) {
            timer->ScheduleAfter(&Event_, delay);
        }

    private:
        void OnSend() {
            Sent_ = true;
        }

    private:
        TMemFuncEvent<TMessage, &TMessage::OnSend> Event_{this};
        bool Sent_{false};
    };

    TTimerWheel timer;

    TMessage hello;
    hello.SendAfter(&timer, TDuration::Minutes(2));
    ASSERT_FALSE(hello.IsSent());

    timer.Advance(TDuration::Minutes(4));
    ASSERT_TRUE(hello.IsSent());
}

TEST(TTimerWheelTest, SequentialScheduleAfter) {
    TTimerWheel timer{TInstant::Zero(), TDuration::MilliSeconds(1)};

    int counter = 0;
    int maxEvents = 10'000;

    std::vector<std::unique_ptr<ITimerEvent>> events;
    events.reserve(maxEvents);

    for (int i = 0; i < maxEvents; i++) {
        events.emplace_back(new TFuncEvent([&counter] {
            counter++;
        }));
        timer.ScheduleAfter(events.back().get(), TDuration::MilliSeconds(i + 1));
    }

    for (int i = 0; i < maxEvents; i++) {
        timer.Advance(TDuration::MilliSeconds(1));
        ASSERT_EQ(counter, i + 1) << "i=" << i;
    }
}

TEST(TTimerWheelTest, SequentialScheduleAt) {
    TTimerWheel timer{TInstant::Zero(), TDuration::MilliSeconds(1)};

    int counter = 0;
    int maxEvents = 10'000;

    std::vector<std::unique_ptr<ITimerEvent>> events;
    events.reserve(maxEvents);

    for (int i = 0; i < maxEvents; i++) {
        events.emplace_back(new TFuncEvent([&counter] {
            counter++;
        }));
        timer.ScheduleAt(events.back().get(), TInstant::MilliSeconds(i + 1));
    }

    for (int i = 0; i < maxEvents; i++) {
        timer.Advance(TDuration::MilliSeconds(1));
        ASSERT_EQ(counter, i + 1) << "i=" << i;
    }
}

TEST(TTimerWheel, Stress) {
    int max = 5'000'000;
    TTimerWheel timer;
    std::vector<std::unique_ptr<TCounter>> counters;

    {
        TFormattedPrecisionTimer measure;
        Cout << "initialize " << max << " events" << Endl;
        counters.reserve(max);
        for (int i = 0; i < max; i++) {
            counters.emplace_back(new TCounter);
        }
    }

    {
        TFormattedPrecisionTimer measure;
        Cout << "schedule all of them" << Endl;
        for (int i = 0; i < max; i++) {
            auto delay = TDuration::MilliSeconds(RandomNumber<ui64>(10 * 60 * 1000) + 1000); // [1s .. 10m)
            timer.ScheduleAfter(counters[i].get(), delay);
        }
    }

    {
        TFormattedPrecisionTimer measure;
        Cout << "advance time: ";
        for (int i = 1; i <= 11; i++) {
            Cout << i << "m  ";
            timer.Advance(TDuration::Minutes(1));
        }
        Cout << Endl;
    }

    {
        TFormattedPrecisionTimer measure;
        Cout << "check that all counters were incremented" << Endl;
        for (int i = 0; i < max; i++) {
            ASSERT_EQ(counters[i]->Value(), 1u) << "in counter[" << i << "]";
        }
    }

    Cout << "done" << Endl;
}
