#pragma once

#include <library/cpp/monlib/metrics/metric_value.h>

namespace NSolomon {

using TCount = ui64;
using TDenom = ui64;

template<typename TMetricValueImpl>
class TAggrPointImpl {
public:
    TAggrPointImpl()
        : Time_{TInstant::Zero()}
        , Denom_{0}
        , Count_{0}
    {}

    template <typename T>
    TAggrPointImpl(TInstant time, T value, ui64 denom, size_t count)
        : Time_{time}
        , Value_{value}
        , Denom_{denom}
        , Count_{count}
    {}

    template <typename T>
    TAggrPointImpl(TInstant time, T value)
        : TAggrPointImpl(time, value, 0, 0)
    {}

    template <typename U = TMetricValueImpl>
    typename std::enable_if_t<std::is_same_v<U, NMonitoring::TMetricValue>,
    TAggrPointImpl&> operator=(const TAggrPointImpl& other) {
        Time_ = other.Time_;
        Value_ = other.Value_;
        Denom_ = other.Denom_;
        Count_ = other.Count_;

        return *this;
    }

    template <typename U = TMetricValueImpl, std::enable_if_t<std::is_same_v<U, NMonitoring::TMetricValueWithType>, int> = 0>
    TAggrPointImpl(TAggrPointImpl&& other)
        : Time_{std::exchange(other.Time_, TInstant::Zero())}
        , Value_{std::move(other.Value_)}
        , Denom_{std::exchange(other.Denom_, 0)}
        , Count_{std::exchange(other.Count_, 0)}
    {}

    template <typename U = TMetricValueImpl, std::enable_if_t<std::is_same_v<U, NMonitoring::TMetricValueWithType>, int> = 0>
    TAggrPointImpl& operator=(TAggrPointImpl&& other) {
        Time_ = std::exchange(other.Time_, TInstant::Zero());
        Value_ = std::move(other.Value_);
        Denom_ = std::exchange(other.Denom_, 0);
        Count_ = std::exchange(other.Count_, 0);

        return *this;
    }

    auto GetTime() const {
        return Time_;
    }

    const TMetricValueImpl& GetValue() const {
        return Value_;
    }

    auto GetCount() const {
        return Count_;
    }

    auto GetDenom() const {
        return Denom_;
    }

    void SetCount(TCount count) {
        Count_ = count;
    };

    void SetDenom(TDenom denom) {
        Denom_ = denom;
    };

    template <typename U = TMetricValueImpl>
    typename std::enable_if_t<std::is_same_v<U, NMonitoring::TMetricValueWithType>,
    NMonitoring::EMetricValueType> GetType() const {
        return Value_.GetType();
    }

    void ClearValue() {
        Time_ = TInstant::Zero();

        if constexpr (std::is_same_v<TMetricValueImpl, NMonitoring::TMetricValueWithType>) {
            Value_.Clear();
        } else {
            Value_ = TMetricValueImpl{ui64{0}};
        }

        Count_ = 0;
        Denom_ = 0;
    }

private:
    TInstant Time_;
    TMetricValueImpl Value_;
    TDenom Denom_;
    TCount Count_;
};

using TAggrPoint = TAggrPointImpl<NMonitoring::TMetricValue>;
using TAggrPointWithType = TAggrPointImpl<NMonitoring::TMetricValueWithType>;

class TAggrTimeSeries  {
public:
    TAggrTimeSeries() = default;

    TAggrTimeSeries(TAggrTimeSeries&& rhs)
        : ValueType_{std::exchange(rhs.ValueType_, NMonitoring::EMetricValueType::UNKNOWN)}
        , AggrPoints_{std::move(rhs.AggrPoints_)}
    {}

    ~TAggrTimeSeries() {
        if (ValueType_ == NMonitoring::EMetricValueType::HISTOGRAM) {
            for (auto& p: AggrPoints_) {
                NMonitoring::SnapshotUnRef<NMonitoring::EMetricValueType::HISTOGRAM>(p);
            }
        } else if (ValueType_ == NMonitoring::EMetricValueType::SUMMARY) {
            for (auto& p: AggrPoints_) {
                NMonitoring::SnapshotUnRef<NMonitoring::EMetricValueType::SUMMARY>(p);
            }
        } else if (ValueType_ == NMonitoring::EMetricValueType::LOGHISTOGRAM) {
            for (auto& p: AggrPoints_) {
                NMonitoring::SnapshotUnRef<NMonitoring::EMetricValueType::LOGHISTOGRAM>(p);
            }
        }

        AggrPoints_.clear();
        ValueType_ = NMonitoring::EMetricValueType::UNKNOWN;
    }

    TAggrTimeSeries& operator=(TAggrTimeSeries&& rhs) {
        ValueType_ = std::exchange(rhs.ValueType_, NMonitoring::EMetricValueType::UNKNOWN);
        AggrPoints_ = std::move(rhs.AggrPoints_);

        return *this;
    }

public:
    void Add(TAggrPointWithType&& point) {
        if (AggrPoints_.empty()) {
            ValueType_ = point.GetType();
        } else {
            CheckTypes(ValueType_, point.GetType());
        }

        auto time = point.GetTime();
        auto count = point.GetCount();
        auto denom = point.GetDenom();

        switch (ValueType_) {
            case NMonitoring::EMetricValueType::DOUBLE: {
                auto value = point.GetValue().AsDouble();
                Add(time, value, denom, count);
                break;
            }
            case NMonitoring::EMetricValueType::UINT64: {
                auto value = point.GetValue().AsUint64();
                Add(time, value, denom, count);
                break;
            }
            case NMonitoring::EMetricValueType::INT64: {
                auto value = point.GetValue().AsInt64();
                Add(time, value, denom, count);
                break;
            }
            case NMonitoring::EMetricValueType::SUMMARY: {
                auto value = point.GetValue().AsSummaryDouble();
                Add(time, value, denom, count);
                break;
            }
            case NMonitoring::EMetricValueType::HISTOGRAM: {
                auto value = point.GetValue().AsHistogram();
                Add(time, value, denom, count);
                break;
            }
            case NMonitoring::EMetricValueType::LOGHISTOGRAM: {
                auto value = point.GetValue().AsLogHistogram();
                Add(time, value, denom, count);
                break;
            }
            case NMonitoring::EMetricValueType::UNKNOWN:
                ythrow yexception() << "unknown value type";
        }

        point.ClearValue();
    }

    template <typename T>
    void Add(TInstant time, T value, ui64 denom, size_t count) {
        if (AggrPoints_.empty()) {
            ValueType_ = NMonitoring::TValueType<T>::Type;
        } else {
            CheckTypes(ValueType_, NMonitoring::TValueType<T>::Type);
        }

        AggrPoints_.emplace_back(time, value, denom, count);

        if (ValueType_ == NMonitoring::EMetricValueType::HISTOGRAM) {
            auto& point = AggrPoints_.back();
            point.GetValue().AsHistogram()->Ref();
        } else if (ValueType_ == NMonitoring::EMetricValueType::SUMMARY) {
            auto& point = AggrPoints_.back();
            point.GetValue().AsSummaryDouble()->Ref();
        } else if (ValueType_ == NMonitoring::EMetricValueType::LOGHISTOGRAM) {
            auto& point= AggrPoints_.back();
            point.GetValue().AsLogHistogram()->Ref();
        }
    }

    template <typename T>
    void Add(TInstant time, T value) {
        Add(time, value, 0, 0);
    }

    template <typename T>
    void AddWithCount(TInstant time, T value, size_t count) {
        Add(time, value, 0, count);
    }

    template <typename T>
    void AddWithDenom(TInstant time, T value, ui64 denom) {
        Add(time, value, denom, 0);
    }

    TAggrPoint& operator[](size_t idx) {
        return AggrPoints_[idx];
    }

    const TAggrPoint& operator[](size_t idx) const {
        return AggrPoints_[idx];
    }

    size_t Size() const {
        return AggrPoints_.size();
    }

    bool Empty() const {
        return AggrPoints_.empty();
    }

    void Clear() noexcept {
        if (ValueType_ == NMonitoring::EMetricValueType::HISTOGRAM) {
            for (auto& p: AggrPoints_) {
                p.GetValue().AsHistogram()->UnRef();
            }
        } else if (ValueType_ == NMonitoring::EMetricValueType::SUMMARY) {
            for (auto& p: AggrPoints_) {
                p.GetValue().AsSummaryDouble()->UnRef();
            }
        } else if (ValueType_ == NMonitoring::EMetricValueType::LOGHISTOGRAM) {
            for (auto& p: AggrPoints_) {
                p.GetValue().AsLogHistogram()->UnRef();
            }
        }

        AggrPoints_.clear();
        ValueType_ = NMonitoring::EMetricValueType::UNKNOWN;
    }

    void SortByTs() {
        NMonitoring::SortPointsByTs(ValueType_, AggrPoints_);
    }

    NMonitoring::EMetricValueType GetValueType() const noexcept {
        return ValueType_;
    }

    void Extend(TAggrTimeSeries&& rhs) {
        if (AggrPoints_.empty()) {
            ValueType_ = rhs.ValueType_;
        } else {
            CheckTypes(ValueType_, rhs.ValueType_);
        }

        static_assert(
            std::is_nothrow_move_constructible_v<TAggrPoint> &&
            std::is_nothrow_move_assignable_v<TAggrPoint>);

        AggrPoints_.insert(
            AggrPoints_.end(),
            std::move_iterator(rhs.AggrPoints_.begin()),
            std::move_iterator(rhs.AggrPoints_.end()));

        rhs.AggrPoints_.clear();
        rhs.ValueType_ = NMonitoring::EMetricValueType::UNKNOWN;
    }

    TAggrTimeSeries Copy() const {
        TAggrTimeSeries copy;
        copy.ValueType_ = ValueType_;
        copy.AggrPoints_ = AggrPoints_;

        return copy;
    }

private:
    static void CheckTypes(NMonitoring::EMetricValueType t1, NMonitoring::EMetricValueType t2) {
        Y_ENSURE(t1 == t2,
                 "TAggrTimeSeries type mismatch: expected " << static_cast<ui32>(t1) <<
                 ", but got " << static_cast<ui32>(t2));
    }

private:
    NMonitoring::EMetricValueType ValueType_ = NMonitoring::EMetricValueType::UNKNOWN;
    TVector<TAggrPoint> AggrPoints_;
};

} // namespace NSolomon
