#include <solomon/libs/cpp/timeseries/timeseries.h>

#include <library/cpp/monlib/metrics/histogram_snapshot.h>
#include <library/cpp/testing/gtest/gtest.h>

namespace NSolomon::NTest {

TEST(TAggrTimeSeriesTest, Construction) {
    TAggrTimeSeries s;

    {
        auto time = TInstant::Zero();
        double value = 1.234;
        ui64 denom = 5;
        size_t count = 6;
        s.Add(time, value, denom, count);

        ASSERT_EQ(s.Size(), 1u);
        const auto& aggrPoint = s[0];

        ASSERT_EQ(aggrPoint.GetTime(), time);
        ASSERT_EQ(aggrPoint.GetValue().AsDouble(), value);
        ASSERT_EQ(aggrPoint.GetDenom(), denom);
        ASSERT_EQ(aggrPoint.GetCount(), count);
    }

    {
        auto time = TInstant::Seconds(123);
        double value = 2.345;
        size_t count = 1;
        ui64 defaultDenom = 0;
        s.AddWithCount(time, value, count);

        ASSERT_EQ(s.Size(), 2u);
        const auto& aggrPoint = s[1];

        ASSERT_EQ(aggrPoint.GetTime(), time);
        ASSERT_EQ(aggrPoint.GetValue().AsDouble(), value);
        ASSERT_EQ(aggrPoint.GetDenom(), defaultDenom);
        ASSERT_EQ(aggrPoint.GetCount(), count);
    }

    {
        auto time = TInstant::Seconds(456);
        double value = 7.987;
        size_t defaultCount = 0;
        ui64 denom = 1000;
        s.AddWithDenom(time, value, denom);

        ASSERT_EQ(s.Size(), 3u);
        const auto& aggrPoint = s[2];

        ASSERT_EQ(aggrPoint.GetTime(), time);
        ASSERT_EQ(aggrPoint.GetValue().AsDouble(), value);
        ASSERT_EQ(aggrPoint.GetDenom(), denom);
        ASSERT_EQ(aggrPoint.GetCount(), defaultCount);
    }
}

TEST(TAggrTimeSeriesTest, SortByTs) {
    auto addPoint = [](TAggrTimeSeries& s, ui64 value) {
        s.Add(TInstant::Seconds(value), value, value, value);
    };

    {
        TAggrTimeSeries s;
        addPoint(s, 5);
        s.Add(TInstant::Zero(), ui64(0));
        addPoint(s, 4);
        addPoint(s, 1);
        addPoint(s, 2);
        addPoint(s, 3);

        size_t oldSize = s.Size();
        s.SortByTs();

        ASSERT_EQ(oldSize, s.Size());

        for (ui64 i = 0; i != s.Size(); ++i) {
            ASSERT_EQ(s[i].GetTime(), TInstant::Seconds(i));
            ASSERT_EQ(s[i].GetValue().AsUint64(), i);
            ASSERT_EQ(s[i].GetCount(), i);
            ASSERT_EQ(s[i].GetDenom(), i);
        }
    }

    {
        TAggrTimeSeries s;

        addPoint(s, 0);
        s.Add(TInstant::Seconds(1), ui64(1));
        s.Add(TInstant::Seconds(1), ui64(2));
        s.Add(TInstant::Seconds(1), ui64(3));
        s.Add(TInstant::Seconds(1), ui64(4));
        addPoint(s, 2);

        ASSERT_EQ(s.Size(), 6u);

        s.SortByTs();

        ASSERT_EQ(s.Size(), 3u);

        auto* p = &s[0];
        ASSERT_EQ(p->GetTime(), TInstant::Zero());
        ASSERT_EQ(p->GetValue().AsUint64(), 0u);
        ASSERT_EQ(p->GetCount(), 0u);
        ASSERT_EQ(p->GetDenom(), 0u);

        p = &s[1];
        ASSERT_EQ(p->GetTime(), TInstant::Seconds(1));
        ASSERT_EQ(p->GetValue().AsUint64(), 4u);
        ASSERT_EQ(p->GetCount(), 0u);
        ASSERT_EQ(p->GetDenom(), 0u);

        p = &s[2];
        ASSERT_EQ(p->GetTime(), TInstant::Seconds(2));
        ASSERT_EQ(p->GetValue().AsUint64(), 2u);
        ASSERT_EQ(p->GetCount(), 2u);
        ASSERT_EQ(p->GetDenom(), 2u);
    }
}

TEST(TAggrPointWithType, General) {
    double value = 1.23;
    auto time = TInstant::Zero();
    TAggrPointWithType pwt{time, value};

    ASSERT_EQ(pwt.GetValue().AsDouble(), value);
    ASSERT_EQ(pwt.GetTime(), time);
    ASSERT_EQ(pwt.GetType(), NMonitoring::EMetricValueType::DOUBLE);

    ASSERT_EQ(pwt.GetCount(), 0u);
    ASSERT_EQ(pwt.GetDenom(), 0u);
    pwt.SetCount(123);
    pwt.SetDenom(456);
    ASSERT_EQ(pwt.GetCount(), 123u);
    ASSERT_EQ(pwt.GetDenom(), 456u);

    ASSERT_EQ(pwt.GetType(), NMonitoring::EMetricValueType::DOUBLE);
}

TEST(TAggrPointWithType, WithTimeSeries) {
    NMonitoring::TBucketBounds bounds{1, 2, 3};
    NMonitoring::TBucketValues values{10, 20, 30};
    auto h = NMonitoring::ExplicitHistogramSnapshot(bounds, values);

    {
        ASSERT_EQ(h.RefCount(), 1u);
        TAggrTimeSeries series;

        {
            TAggrPointWithType point{TInstant::Zero(), h.Get()};
            ASSERT_EQ(h.RefCount(), 2u);

            series.Add(std::move(point));
            ASSERT_EQ(h.RefCount(), 2u);
        }

        ASSERT_EQ(h.RefCount(), 2u);
    }

    ASSERT_EQ(h.RefCount(), 1u);
    ASSERT_EQ(h->Count(), 3u);

    for (size_t i = 0; i != h->Count(); ++i) {
        auto j = i + 1;
        ASSERT_EQ(h->UpperBound(i), j);
        ASSERT_EQ(h->Value(i), j * 10);
    }
}

} // namespace NSolomon::NTest
