#pragma once

#include "trace_fwd.h"

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/actors/events/events.h>

#include <util/system/src_location.h>

namespace NSolomon::NTracing {

struct TSpan {
    ui64 Id{0};
    ui64 ParentId{0};
    TInstant Begin{TInstant::Zero()};
    TInstant End{TInstant::Zero()};
    TString Description;
};

class TTraceId {
public:
    TTraceId() {
        Id_.fill(0);
    }
    static TTraceId From(const TSpanId& span) {
        TTraceId ret;
        static_assert(sizeof(ret.Id_) == NWilson::TTraceId::GetTraceIdSize());
        memcpy(ret.Id_.data(), span.GetTraceIdPtr(), span.GetTraceIdSize());
        return ret;
    }
    TTraceId(const TTraceId& other) = default;
    TTraceId(TTraceId&& other) = default;
    TTraceId& operator=(const TTraceId& other) = default;
    TTraceId& operator=(TTraceId&& other) = default;
    bool operator<(const TTraceId& other) const {
        return Id_ < other.Id_;
    }
    bool operator==(const TTraceId& other) const {
        return Id_ == other.Id_;
    }
    bool Empty() const {
        return Id_[0] == 0 && Id_[1] == 0;
    }
    ui64 High() const {
        return Id_[0];
    }
    ui64 Low() const {
        return Id_[1];
    }
    TString Hex() const;
    template <typename H>
    friend H AbslHashValue(H h, const TTraceId& o) {
        return H::combine(std::move(h), o.Id_);
    }
private:
    // Implementation detail of NWilson::TTraceId::TTrace
    std::array<ui64, 2> Id_;
};

inline ui64 GetSpanId(const TSpanId& span) {
    ui64 spanId;
    static_assert(sizeof(spanId) == NWilson::TTraceId::GetSpanIdSize());
    memcpy(&spanId, span.GetSpanIdPtr(), span.GetSpanIdSize());
    return spanId;
}

inline TInstant TraceingClockNow() {
    return TInstant::MicroSeconds(NActors::GetMonotonicMicroSeconds());
}

class TTracingEvents: private TEventSlot<EEventSpace::Libs, ELibSlot::Tracing> {
    enum {
        ReportBegin = SpaceBegin,
        ReportEnd,
        ReportMissing,
        GetTraceReq,
        GetTraceResp,
        EraseTraceReq,
        End,
    };

    static_assert(End < SpaceEnd, "too many event types");
public:
    template <typename TDerived, ui32 Event>
    struct TTracingEvent: public NActors::TEventLocal<TDerived, Event> {
        using TBase = TTracingEvent<TDerived, Event>;

        TSourceLocation Loc;
        const char* Function;
        TInstant Moment;

        TTracingEvent(TSourceLocation loc, const char* function) noexcept
            : Loc(loc)
            , Function(function)
            , Moment(TraceingClockNow())
        { }
    };

    struct TEvReportBegin: public TTracingEvent<TEvReportBegin, ReportBegin> {
        TSpanId ParentSpan;
        TString Description;

        TEvReportBegin(TSourceLocation loc, const char* function, TSpanId parentSpan, TString&& description) noexcept
                : TBase{loc, function}
                , ParentSpan(std::move(parentSpan))
                , Description(std::move(description))
        { }
    };

    struct TEvReportEnd: public TTracingEvent<TEvReportEnd, ReportEnd> {
        using TBase::TTracingEvent;
    };

    struct TEvReportMissing: public TTracingEvent<TEvReportMissing, ReportMissing> {
        bool Starting;

        TEvReportMissing(TSourceLocation loc, const char* function, bool starting) noexcept
                : TBase{loc, function}
                , Starting(starting)
        { }
    };

    struct TEvGetTraceRequest: public NActors::TEventLocal<TEvGetTraceRequest, GetTraceReq> {
        TTraceId ReqTraceId;

        TEvGetTraceRequest(TTraceId traceId) noexcept
                : ReqTraceId(std::move(traceId))
        { }
    };

    struct TEvGetTraceResponse: public NActors::TEventLocal<TEvGetTraceResponse, GetTraceResp> {
        TTraceId TraceId;
        TVector<TSpan> Spans;

        TEvGetTraceResponse(TTraceId traceId, TVector<TSpan> spans) noexcept
                : TraceId{std::move(traceId)}
                , Spans{std::move(spans)}
        { }
    };

    struct TEvEraseTraceRequest: public NActors::TEventLocal<TEvEraseTraceRequest, EraseTraceReq> {
        TTraceId ReqTraceId;

        TEvEraseTraceRequest(TTraceId traceId) noexcept
                : ReqTraceId(std::move(traceId))
        { }
    };
};

void InitTracingService(NMonitoring::TMetricRegistry& registry, NActors::TActorSystem& actorSystem, ui32 executorPool, const TString& service);

inline NActors::TActorId TracingServiceId() {
    return NActors::TActorId(0, TStringBuf("TracingServ\0", 12));
}

inline void EraseTrace(TTraceId traceId) {
    Y_VERIFY(NActors::TlsActivationContext, "trying to trace outside actor system");
    NActors::TActorContext::ActorSystem()->Send(TracingServiceId(), new TTracingEvents::TEvEraseTraceRequest{std::move(traceId)});
}


namespace NPrivate {

inline TSpanId CreateSpanAndReportBegin(const TSourceLocation loc, const char* function, const TSpanId& parentSpan, ui8 verbosity, TString&& description) {
    TSpanId span = parentSpan.Span(verbosity);

    Y_VERIFY(NActors::TlsActivationContext, "trying to trace outside actor system");
    NActors::TActorContext::ActorSystem()->Send(new NActors::IEventHandle{
        TracingServiceId(), {},
        new TTracingEvents::TEvReportBegin{loc, function, TSpanId(parentSpan), std::move(description)},
        0, 0, nullptr, TSpanId(span)
    });

    return span;
}

inline void ReportSpanEnd(const TSourceLocation loc, const char* function, const TSpanId& span) {
    Y_VERIFY(NActors::TlsActivationContext, "trying to trace outside actor system");
    NActors::TActorContext::ActorSystem()->Send(new NActors::IEventHandle{
            TracingServiceId(), {},
            new TTracingEvents::TEvReportEnd{loc, function},
            0, 0, nullptr, TSpanId(span)
    });
}

inline void ReportMissingTraceContext(const TSourceLocation loc, const char* function, bool starting) {
    Y_VERIFY(NActors::TlsActivationContext, "trying to trace outside actor system");
    NActors::TActorContext::ActorSystem()->Send(TracingServiceId(),
            new TTracingEvents::TEvReportMissing{loc, function, starting});
}

} // NPrivate

} // NSolomon::NTracing

// When 1 any traceId == 0 is reported as an error (i.e. traceId:spanId was not passed properly)
// Don't use when sampling rate is not 100, otherwise each span that was sampled out will be reported
#define DEBUG_LOST_TRACES 0

#if defined(NDEBUG) && DEBUG_LOST_TRACES
# error "Debugging traces enabled for release build"
#endif

#if DEBUG_LOST_TRACES
# define _REPORT_MISSING_TRACE_CONTEXT(starting) \  // NOLINT(readability-identifier-naming)
    ::NSolomon::NTracing::NPrivate::ReportMissingTraceContext(__LOCATION__, __FUNCTION__, starting)
#else
# define _REPORT_MISSING_TRACE_CONTEXT(starting) do {} while(false)  // NOLINT(readability-identifier-naming)
#endif

// TODO: support span verbosity
#define TRACING_NEW_SPAN_START(traceCtx, details) [_traceCtx(traceCtx), msg = TStringBuilder{} << details]() mutable { \
            if (!_traceCtx) { \
                _REPORT_MISSING_TRACE_CONTEXT(true); \
                return ::NSolomon::NTracing::TSpanId{}; \
            }; \
            return ::NSolomon::NTracing::NPrivate::CreateSpanAndReportBegin(__LOCATION__, __FUNCTION__, \
                    _traceCtx, 0, std::move(msg)); \
        }()

#define TRACING_SPAN_END(span) do { \
            const auto& _span = (span); \
            if (_span) { ::NSolomon::NTracing::NPrivate::ReportSpanEnd(__LOCATION__, __FUNCTION__, _span); } \
            else { _REPORT_MISSING_TRACE_CONTEXT(false); } \
        } while (false)

#define TRACING_SPAN_END_EV(ev) TRACING_SPAN_END(ev->TraceId)
