#include "udp_sender.h"

#include <solomon/libs/cpp/logging/logging.h>

#include <util/network/socket.h>

namespace NSolomon::NTracing {

namespace NPrivate {

class NoopSender: public IUdpSender {
public:
    NoopSender() {
        MON_INFO(Tracing, "Reporting spans is disabled");
    }
    ~NoopSender() override = default;
    bool Send(const char*, int) override {
        return true;
    }
};

class TUdpSender: public IUdpSender {
public:
    TUdpSender(ui16 port) {
        Sock_ = socket(AF_INET6, SOCK_DGRAM, IPPROTO_UDP);
        Y_ENSURE(Sock_ != INVALID_SOCKET, "UDP socket creation failed");

        SetNonBlock(Sock_);

        sockaddr_in6 clientAddr;
        Zero(clientAddr);
        clientAddr.sin6_family = AF_INET6;
        clientAddr.sin6_addr = in6addr_any;
        Y_ENSURE(bind(Sock_, reinterpret_cast<const sockaddr*>(&clientAddr), sizeof(clientAddr)) == 0, "failed to bind to a local address");

        sockaddr_in6 agentAddr;
        Zero(agentAddr);
        agentAddr.sin6_family = AF_INET6;
        agentAddr.sin6_addr = in6addr_loopback;
        agentAddr.sin6_port = htons(port);

        Y_ENSURE(connect(Sock_, reinterpret_cast<const sockaddr*>(&agentAddr), sizeof(agentAddr)) == 0, "failed to connect to agent");
        MON_INFO(Tracing, "Reporting spans to [::1]:" << port << " by UDP");
    }

    ~TUdpSender() = default;

    bool Send(const char* buf, int size) override {
        int ret = ::send(Sock_, buf, size, 0);

        if (ret < 0) {
            MON_ERROR(Tracing, "failed to send: " << WSAGetLastError());
            return false;
        }

        if (ret < size) {
            MON_ERROR(Tracing, "partial send, only " << ret << " bytes out of " << size);
            return false;
        }

        return true;
    }
private:
    SOCKET Sock_;
};

}

std::unique_ptr<IUdpSender> MakeUdpSender(ui16 port) {
    if (port == 0) {
        return std::make_unique<NPrivate::NoopSender>();
    }
    return std::make_unique<NPrivate::TUdpSender>(port);
}

}
