#include <solomon/libs/cpp/ts_codec/bit_reader.h>

#include <util/generic/size_literals.h>

#include <benchmark/benchmark.h>

#include <type_traits>

using namespace NSolomon::NTs;

enum class EAlignment {
    Aligned,
    Unaligned,
};

template <EAlignment Alignment, typename BenchFn>
void RunBenchmark(benchmark::State& state, BenchFn fn) {
    TBuffer buffer;
    buffer.Fill('A', 16_MB);
    TBitReader reader{buffer.data(), buffer.size() * 8};

    if constexpr (Alignment == EAlignment::Unaligned) {
        benchmark::DoNotOptimize(reader.ReadBit());
        benchmark::DoNotOptimize(reader.ReadBit());
    }

    size_t initPos = reader.Pos();
    for (auto _: state) {
        benchmark::DoNotOptimize(fn(reader));

        if (reader.Left() < 100) {
            reader.SetPos(initPos);
        }
    }

    using TResult = decltype(fn(reader));
    size_t bytesProcessed = (std::is_same_v<TResult, bool>)
            ? state.iterations() / 8
            : state.iterations() * sizeof(TResult);

    state.SetBytesProcessed(bytesProcessed);
    state.SetItemsProcessed(state.iterations());
}

static void BmReadBit(benchmark::State& state) {
    RunBenchmark<EAlignment::Aligned>(state, [](TBitReader& reader) -> bool {
        return reader.ReadBit();
    });
}

template <EAlignment Alignment>
static void BmReadInt8(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadInt8();
    });
}

template <EAlignment Alignment>
static void BmReadInt32(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadInt32();
    });
}

template <EAlignment Alignment>
static void BmReadInt64(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadInt64();
    });
}

template <EAlignment Alignment>
static void BmReadDouble(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadDouble();
    });
}

template <EAlignment Alignment>
static void BmReadInt8Bits(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadInt8(8);
    });
}

template <EAlignment Alignment>
static void BmReadInt32Bits(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadInt32(32);
    });
}

template <EAlignment Alignment>
static void BmReadInt64Bits(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadInt64(64);
    });
}

template <EAlignment Alignment>
static void BmReadVarInt32(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadVarInt32();
    });
}

template <EAlignment Alignment>
static void BmReadVarInt64(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadVarInt64();
    });
}

template <EAlignment Alignment>
static void BmReadVarInt32Mode(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadVarInt32Mode();
    });
}

template <EAlignment Alignment>
static void BmReadVarInt64Mode(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadVarInt64Mode();
    });
}

template <EAlignment Alignment>
static void BmReadOnes(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](TBitReader& reader) {
        return reader.ReadOnes(8);
    });
}

BENCHMARK(BmReadBit);

BENCHMARK_TEMPLATE(BmReadInt8, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadInt8, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadInt32, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadInt32, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadInt64, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadInt64, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadDouble, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadDouble, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadInt8Bits, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadInt8Bits, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadInt32Bits, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadInt32Bits, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadInt64Bits, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadInt64Bits, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadVarInt32, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadVarInt32, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadVarInt64, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadVarInt64, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadVarInt32Mode, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadVarInt32Mode, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadVarInt64Mode, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadVarInt64Mode, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmReadOnes, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmReadOnes, EAlignment::Unaligned);
