#include <solomon/libs/cpp/ts_codec/bit_writer.h>

#include <util/generic/size_literals.h>

#include <benchmark/benchmark.h>

#include <cmath>

using namespace NSolomon::NTs;

enum class EAlignment {
    Aligned,
    Unaligned,
};

template <EAlignment Alignment, typename BenchFn>
void RunBenchmark(benchmark::State& state, BenchFn fn) {
    TBitBuffer buffer{4_MB};
    TBitWriter w{&buffer};

    if constexpr (Alignment == EAlignment::Unaligned) {
        w.WriteBit(true);
        w.WriteBit(false);
    }

    fn(state, w);

    state.SetItemsProcessed(state.iterations());
    state.SetBytesProcessed(ByteCount(w.Pos()));
}

static void BmWriteBit(benchmark::State& state) {
    RunBenchmark<EAlignment::Aligned>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteBit(false);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteInt8(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteInt8(0x42);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteInt32(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteInt32(0xdeadbeef);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteInt64(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteInt64(0xdeadbeef'deadbeefull);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteDouble(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteDouble(M_PI);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteInt8Bits(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteInt8(0xbe, 8);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteInt32Bits(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteInt32(0xdeadbeef, 32);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteInt64Bits(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteInt64(0xdeadbeef'deadbeefull, 64);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteVarInt32(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        ui32 value = static_cast<ui32>(state.range(0));
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteVarInt32(value);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteVarInt64(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        ui64 value = static_cast<ui64>(state.range(0));
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteVarInt64(value);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteVarInt32Mode(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        ui32 value = static_cast<ui32>(state.range(0));
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteVarInt32Mode(value);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteVarInt64Mode(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        ui64 value = static_cast<ui64>(state.range(0));
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteVarInt64Mode(value);
        }
    });
}

template <EAlignment Alignment>
static void BmWriteOnes(benchmark::State& state) {
    RunBenchmark<Alignment>(state, [](auto& state, auto& writer) {
        ui64 value = static_cast<ui64>(state.range(0));
        for (auto it = state.begin(), end = state.end(); it != end; ++it) {
            writer.WriteOnes(value, 8);
        }
    });
}

BENCHMARK(BmWriteBit);

BENCHMARK_TEMPLATE(BmWriteInt8, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmWriteInt8, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmWriteInt32, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmWriteInt32, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmWriteInt64, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmWriteInt64, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmWriteDouble, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmWriteDouble, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmWriteInt8Bits, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmWriteInt8Bits, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmWriteInt32Bits, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmWriteInt32Bits, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmWriteInt64Bits, EAlignment::Aligned);
BENCHMARK_TEMPLATE(BmWriteInt64Bits, EAlignment::Unaligned);

BENCHMARK_TEMPLATE(BmWriteVarInt32, EAlignment::Aligned)->RangeMultiplier(32)->Range(64, 1<<20);
BENCHMARK_TEMPLATE(BmWriteVarInt32, EAlignment::Unaligned)->RangeMultiplier(32)->Range(64, 1<<20);

BENCHMARK_TEMPLATE(BmWriteVarInt64, EAlignment::Aligned)->RangeMultiplier(32)->Range(64, 1<<20);
BENCHMARK_TEMPLATE(BmWriteVarInt64, EAlignment::Unaligned)->RangeMultiplier(32)->Range(64, 1<<20);

BENCHMARK_TEMPLATE(BmWriteVarInt32Mode, EAlignment::Aligned)->RangeMultiplier(32)->Range(64, 1<<20);
BENCHMARK_TEMPLATE(BmWriteVarInt32Mode, EAlignment::Unaligned)->RangeMultiplier(32)->Range(64, 1<<20);

BENCHMARK_TEMPLATE(BmWriteVarInt64Mode, EAlignment::Aligned)->RangeMultiplier(32)->Range(64, 1<<20);
BENCHMARK_TEMPLATE(BmWriteVarInt64Mode, EAlignment::Unaligned)->RangeMultiplier(32)->Range(64, 1<<20);

BENCHMARK_TEMPLATE(BmWriteOnes, EAlignment::Aligned)->DenseRange(0, 8, 2);
BENCHMARK_TEMPLATE(BmWriteOnes, EAlignment::Unaligned)->DenseRange(0, 8, 2);
