#include <solomon/libs/cpp/ts_codec/double_ts_codec.h>

#include <util/generic/size_literals.h>
#include <util/random/random.h>

#include <benchmark/benchmark.h>

using namespace NSolomon::NTs;

constexpr TColumnSet Columns{EColumn::TS, EColumn::VALUE};
constexpr size_t PointCount = 1000;

std::vector<TDoublePoint> Generate(size_t count, const std::function<double(size_t, double)>& gen) {
    std::vector<TDoublePoint> points;
    points.reserve(count);

    TInstant now = TInstant::Now();
    TDoublePoint point;

    for (size_t i = 0; i < count; ++i) {
        point.Time = now;
        point.Num = gen(i, point.Num);
        points.emplace_back(point);

        now += TDuration::Seconds(15);
    }
    return points;
}

// see https://a.yandex-team.ru/arc/trunk/arcadia/solomon/libs/java/ts-model/src/point/column/ValueRandomData.java
double RandomNum() {
    auto c = RandomNumber(12u);
    switch (c) {
        case 0u: return RandomNumber<bool>() ? 0 : 1;
        case 1u: return RandomNumber<ui32>();
        case 2u: return RandomNumber<ui32>(200) - 100;
        case 3u: return RandomNumber<double>();
        case 4u: return BitCast<double>(RandomNumber<ui64>());
        case 5u: return std::numeric_limits<double>::quiet_NaN();
        case 6u: return std::numeric_limits<double>::infinity();
        case 7u: return -std::numeric_limits<double>::infinity();
        case 8u: return std::numeric_limits<double>::max();
        case 9u: return std::numeric_limits<double>::min();
        case 10u: return BitCast<double>(RandomNumber<ui64>(5));
        case 11u: return BitCast<double>(ui64(1) << RandomNumber<ui8>(64));;
        default:
            Y_FAIL("unsupported case %d", c);
    }
};

static void Encode(benchmark::State& state, const std::vector<TDoublePoint>& points) {
    TBitBuffer buffer{TBuffer{4_MB}};
    for (auto _: state) {
        buffer.Clear();
        TBitWriter w{&buffer};
        auto encoder = TDoubleTsEncoder{Columns, &w};
        for (const auto& point: points) {
            encoder.EncodePoint(point);
        }
        benchmark::DoNotOptimize(buffer);
    }
}

static void Decode(benchmark::State& state, const std::vector<TDoublePoint>& points) {
    TBitBuffer buffer{TBuffer{4_MB}};
    {
        TBitWriter w{&buffer};
        TDoubleTsEncoder encoder{Columns, &w};
        for (const auto& point: points) {
            encoder.EncodePoint(point);
        }
        encoder.Flush();
    }

    for (auto _: state) {
        TDoubleTsDecoder decoder{Columns, buffer};
        TDoublePoint point;
        while (decoder.NextPoint(&point)) {
            // nop
        }
        benchmark::DoNotOptimize(point);
    }
}

static void BmEncodeSeq(benchmark::State& state) {
    auto points = Generate(PointCount, [](size_t, double prev) { return prev + 1.0; });
    Encode(state, points);
}

static void BmEncodeRandom(benchmark::State& state) {
    auto points = Generate(PointCount, [](size_t, double) { return RandomNum(); });
    Encode(state, points);
}

static void BmEncodeRandomGrow(benchmark::State& state) {
    auto points = Generate(PointCount, [](size_t, double prev) {
        return prev + static_cast<i64>(RandomNumber<ui64>());
    });
    Encode(state, points);
}

static void BmDecodeSeq(benchmark::State& state) {
    auto points = Generate(PointCount, [](size_t, double prev) { return prev + 1.0; });
    Decode(state, points);
}

static void BmDecodeRandom(benchmark::State& state) {
    auto points = Generate(PointCount, [](size_t, double) { return RandomNum(); });
    Decode(state, points);
}

static void BmDecodeRandomGrow(benchmark::State& state) {
    auto points = Generate(PointCount, [](size_t, double prev) {
        return prev + static_cast<i64>(RandomNumber<ui64>());
    });
    Encode(state, points);
}

BENCHMARK(BmEncodeSeq);
BENCHMARK(BmEncodeRandom);
BENCHMARK(BmEncodeRandomGrow);

BENCHMARK(BmDecodeSeq);
BENCHMARK(BmDecodeRandom);
BENCHMARK(BmDecodeRandomGrow);
