#pragma once

#include "bit_span.h"

#include <util/generic/buffer.h>
#include <util/generic/yexception.h>

namespace NSolomon::NTs {

/**
 * Template of specific storage operations for TBitStorage.
 */
template <typename T>
struct TStorageOps {
    constexpr static void* Data(T& storage);
    constexpr static const void* Data(const T& storage);
    constexpr static size_t Size(const T& storage);
    constexpr static size_t Capacity(const T& storage);
    static void Resize(T& storage, size_t size);
    static void Clear(T& storage);
};

/**
 * Template of bit storage that can use different containers for store bits.
 */
template <typename TContainer, typename TOps = TStorageOps<TContainer>>
class TBitStorage {
public:
    /**
     * Constructs a bit storage using the characters from {@code str}.
     * Valid characters are {' ', '0', '1'}.
     *
     * @param str  string of bits
     * @return new bit storage
     */
    static TBitStorage<TContainer, TOps> FromString(std::string_view str);

    /**
     * Constructs empty bit storage.
     */
    TBitStorage()
        : Size_{0}
    {
    }

    /**
     * Constructs bit storage which will use given container to store bits.
     */
    explicit TBitStorage(TContainer&& storage) noexcept
        : Container_{std::forward<TContainer>(storage)}
        , Size_{TOps::Size(Container_) * BitsSize<ui8>()}
    {
    }

    explicit TBitStorage(TBitSpan data)
        : Size_{data.Size()}
    {
        TOps::Resize(Container_, data.SizeBytes());
        std::memcpy(TOps::Data(Container_), data.Data(), data.SizeBytes());
    }

    /**
     * Returns a pointer to the underlying bytes array.
     *
     * @return a pointer to the underlying bytes array.
     */
    constexpr ui8* Data() noexcept {
        return reinterpret_cast<ui8*>(TOps::Data(Container_));
    }

    /**
     * Returns a constant pointer to the underlying bytes array.
     *
     * @return a constant pointer to the underlying bytes array.
     */
    constexpr const ui8* Data() const noexcept {
        return reinterpret_cast<const ui8*>(TOps::Data(Container_));
    }

    /**
     * Returns the number of bits in this storage.
     *
     * @return number of bits
     */
    constexpr size_t Size() const noexcept {
        return Size_;
    }

    /**
     * Returns the number of bytes in this storage.
     *
     * @return number of bytes
     */
    constexpr size_t SizeBytes() const noexcept {
        return ByteCount(Size_);
    }

    /**
     * Returns the number of bits that the storage has currently allocated space for.
     * Returned value always rounded up to 8 bits.
     *
     * @return number of bits
     */
    constexpr size_t Capacity() const noexcept {
        return TOps::Capacity(Container_) * BitsSize<ui8>();
    }

    /**
     * Returns the number of bytes allocated for this storage.
     *
     * @return number of bytes
     */
    constexpr size_t CapacityBytes() const noexcept {
        return TOps::Capacity(Container_);
    }

    /**
     * Implicitly converts buffer to a bit span
     * @return new bit span
     */
    constexpr operator TBitSpan() const noexcept {
        return {Data(), Size()};
    }

    /**
     * Returns value of idx-th bit of the bit storage.
     *
     * @param idx  bit index from 0
     * @return corresponding bit value (not a reference)
     */
    constexpr bool operator[](size_t idx) const noexcept {
        Y_VERIFY_DEBUG(idx < Size());
        auto [byte, bit] = SplitBitIndex(idx);
        return Data()[byte] & (1u << bit);
    }

    /**
     * Appends the given bit to the end of the storage
     * @param value  bit value
     */
    void PushBack(bool value) {
        auto [byte, bit] = SplitBitIndex(Size_++);

        if (byte >= TOps::Size(Container_)) {
            TOps::Resize(Container_, byte + 1);
        }

        if (value) {
            Data()[byte] |= ui8(1) << bit;
        } else {
            Data()[byte] &= ~(ui8(1) << bit);
        }
    }

    /**
     * Resizes the storage to contain the given number of bits.
     * If the current size is less than {@code newSize}, additional '0' bits are appended.
     * If the current size is greater than {@code newSize}, the storage is reduced to a new size.
     *
     * @param newSize new size in bits
     */
    void Resize(size_t newSize) {
        TOps::Resize(Container_, ByteCount(newSize));
        Size_ = newSize;
    }

    /**
     * Clears underlying container and resets size of bit storage.
     */
    void Clear() {
        TOps::Clear(Container_);
        Size_ = 0;
    }

    /**
     * Returns constant reference to the underlying container.
     *
     * @return constant reference to the underlying container.
     */
    const TContainer& Container() const & {
        return Container_;
    }

    /**
     * Returns underlying container by moving it from the storage.
     *
     * @return underlying container.
     */
    TContainer Container() && {
        return std::move(Container_);
    }

    /**
     * Compares content of two bit storages.
     *
     * @param lhs  left hand side bit storage
     * @param rhs  right hand side bit storage
     * @return true iff content of both bit storages are identical, false otherwise
     */
    template <typename T, typename U>
    friend bool operator==(const TBitStorage<T, U>& lhs, const TBitStorage<T, U>& rhs);

    /**
     * Pretty print storage fields (without storage content).
     *
     * @param out     output stream
     * @param span    span value
     * @return output stream
     */
    template <typename T, typename U>
    friend std::ostream& operator<<(std::ostream& out, const TBitStorage<T, U>& buf);

private:
    TContainer Container_;
    size_t Size_;
};

template<typename TContainer, typename TOps>
TBitStorage<TContainer, TOps> TBitStorage<TContainer, TOps>::FromString(std::string_view str) {
    TBitStorage<TContainer, TOps> buf;
    for (char ch: str) {
        if (ch == ' ') {
            continue;
        } else if (ch == '0') {
            buf.PushBack(false);
        } else if (ch == '1') {
            buf.PushBack(true);
        } else {
            ythrow yexception() << "unexpected char: " << ch;
        }
    }
    return buf;
}

template <typename T, typename U>
bool operator==(const TBitStorage<T, U>& lhs, const TBitStorage<T, U>& rhs) {
    return static_cast<TBitSpan>(lhs) == static_cast<TBitSpan>(rhs);
}

template <typename T, typename U>
std::ostream& operator<<(std::ostream& out, const TBitStorage<T, U>& buf) {
    using namespace std::literals;

    return out << "{Data="sv << buf.Data() << ", Size="sv << buf.Size_ << '}';
}

template <>
struct TStorageOps<TBuffer> {
    constexpr static void* Data(TBuffer& storage) {
        return storage.data();
    }

    constexpr static const void* Data(const TBuffer& storage) {
        return storage.data();
    }

    constexpr static size_t Capacity(const TBuffer& storage) {
        return storage.Capacity();
    }

    constexpr static size_t Size(const TBuffer& storage) {
        return storage.Size();
    }

    static void Resize(TBuffer& storage, size_t size) {
        size_t oldSize = storage.Size();
        storage.Resize(size);
        if (size > oldSize) {
            std::memset(storage.Data() + oldSize, '\0', size - oldSize);
        }
    }

    static void Clear(TBuffer& storage) {
        storage.Clear();
    }
};

using TBitBuffer = TBitStorage<TBuffer>;

template <>
struct TStorageOps<TString> {
    constexpr static void* Data(TString& storage) {
        return storage.begin();
    }

    constexpr static const void* Data(const TString& storage) {
        return storage.data();
    }

    constexpr static size_t Capacity(const TString& storage) {
        return storage.capacity();
    }

    constexpr static size_t Size(const TString& storage) {
        return storage.size();
    }

    static void Resize(TString& storage, size_t size) {
        storage.resize(size, '\0');
    }

    static void Clear(TString& storage) {
        storage.clear();
    }
};

using TBitString = TBitStorage<TString>;

} // namespace NSolomon::NTs
