#include "bit_reader.h"

namespace NSolomon::NTs {
namespace {

constexpr ui32 MaxVarInt64Bytes = 10;
constexpr size_t BitsPerMode32[] = { 0, 4, 8, 12, 32 };
constexpr size_t BitsPerMode64[] = { 0, 4, 8, 12, 16, 24, 32, 64 };

// TODO: implement fast varint parsing (see stockpile_codec library)

std::optional<ui64> ReadVarInt64Slow(TBitReader* r) {
    ui64 result = 0;
    ui32 count = 0;
    ui8 b;

    do {
        if (Y_UNLIKELY(count == MaxVarInt64Bytes || r->Left() < BitsSize<ui8>())) {
            return std::nullopt;
        }

        b = r->ReadInt8();
        result |= static_cast<ui64>(b & 0x7f) << (7 * count);
        ++count;
    } while (b & 0x80);

    return result;
}

} // namespace

std::optional<ui32> TBitReader::ReadVarInt32() noexcept {
    auto result = ReadVarInt64Slow(this);
    if (Y_LIKELY(result && *result <= std::numeric_limits<ui32>::max())) {
        return static_cast<ui32>(*result);
    }
    return std::nullopt;
}

std::optional<ui64> TBitReader::ReadVarInt64() noexcept {
    return ReadVarInt64Slow(this);
}

std::optional<ui64> TBitReader::ReadVarInt32Mode() noexcept {
    if (Y_UNLIKELY(Left() == 0)) {
        return std::nullopt;
    }

    ui8 mode = ReadOnes(std::size(BitsPerMode32));
    size_t bits = BitsPerMode32[mode];
    return Y_UNLIKELY(Left() < bits) ? std::nullopt : std::optional{ReadInt32(bits)};
}

std::optional<ui64> TBitReader::ReadVarInt64Mode() noexcept {
    if (Y_UNLIKELY(Left() == 0)) {
        return std::nullopt;
    }

    ui8 mode = ReadOnes(std::size(BitsPerMode64));
    size_t bits = BitsPerMode64[mode];
    return Y_UNLIKELY(Left() < bits) ? std::nullopt : std::optional{ReadInt64(bits)};
}

ui8 TBitReader::ReadOnes(ui8 max) noexcept {
    Y_VERIFY_DEBUG(max <= BitsSize<ui8>());
    Y_VERIFY_DEBUG(Left() > 0);

    ui8 count = 0;
    while (count != max && ReadBit()) {
        ++count;
    }
    return count;
}

} // namespace NSolomon::NTs
