#pragma once

#include "bits.h"

#include <util/generic/strbuf.h>

namespace NSolomon::NTs {

/**
 * Reference to contiguous block of memory with bit-level size.
 * Current implementation considers that span starts from byte aligned memory address.
 */
class TBitSpan {
public:
    /**
     * Constructs an empty span whose {@code Data() == nullptr} and {@code Size() == 0}.
     */
    constexpr TBitSpan() noexcept
        : Data_{nullptr}
        , Size_{0}
    {
    }

    /**
     * Constructs a span that is a view over the memory block [data, data + (size + 7) / 8].
     *
     * @param data  address of memory block
     * @param size  size in bits
     */
    constexpr TBitSpan(const void* data, size_t size) noexcept
        : Data_{static_cast<const ui8*>(data)}
        , Size_{size}
    {
    }

    /**
     * Constructs a span that is a view over the string buffer.
     *
     * @param sb  string buffer
     */
    constexpr explicit TBitSpan(TStringBuf sb) noexcept
        : TBitSpan(sb.data(), sb.size() * BitsSize<char>())
    {
    }

    /**
     * Defaulted copy constructor copies the size and data pointer.
     */
    constexpr TBitSpan(const TBitSpan&) noexcept = default;

    /**
     * Defaulted assignment operator performs a shallow copy of the data pointer and the size.
     */
    constexpr TBitSpan& operator=(const TBitSpan&) noexcept = default;

    /**
     * @return a pointer to the beginning of the span memory.
     */
    constexpr const ui8* Data() const noexcept {
        return Data_;
    }

    /**
     * @return a size of the span in bits.
     */
    constexpr size_t Size() const noexcept {
        return Size_;
    }

    /**
     * @return a size of the span in bytes.
     */
    constexpr size_t SizeBytes() const noexcept {
        return ByteCount(Size_);
    }

    /**
     * Returns value of idx-th bit of the span.
     *
     * @param idx  bit index from 0
     * @return corresponding bit value (not a reference)
     */
    constexpr bool operator[](size_t idx) const noexcept {
        auto [byte, bit] = SplitBitIndex(idx);
        return Data_[byte] & (1u << bit);
    }

    /**
     * Compares content of two spans.
     *
     * @param rhs  right hand side span
     * @return true iff content of both spans are identical or both spans
     *         created over the same memory block, false otherwise
     */
    bool operator==(const TBitSpan& rhs) const noexcept;

    /**
     * Pretty print span fields.
     *
     * @param out     output stream
     * @param span    span value
     * @return output stream
     */
    friend std::ostream& operator<<(std::ostream& out, TBitSpan span);

private:
    const ui8* Data_;
    size_t Size_;
};

} // namespace NSolomon::NTs
