#pragma once

#include <util/system/types.h>

#include <limits>
#include <utility>

namespace NSolomon::NTs {

/**
 * @return size of T in bits
 */
template <typename T>
constexpr size_t BitsSize() {
    return sizeof(T) << 3;
}

/**
 * @return true if bit N in value is set.
 */
template <unsigned N, typename T>
constexpr bool BitTest(T value) noexcept {
    static_assert(N <= sizeof(value) * 8);
    return (value & (static_cast<T>(1) << N)) != 0;
}

/**
 * Set bit N in value.
 */
template <unsigned N, typename T>
constexpr T BitSet(T value) noexcept {
    static_assert(N <= sizeof(value) * 8);
    return value | (static_cast<T>(1) << N);
}

/**
 * Set bit N in value.
 */
template <unsigned N, typename T>
constexpr T BitClear(T value) noexcept {
    static_assert(N <= sizeof(value) * 8);
    return value & ~(static_cast<T>(1) << N);
}

/**
 * Convert bit index to byte index and bit offset in that byte.
 *
 * @param index  bit index
 * @return tuple { byteIndex, bitOffset }
 */
constexpr std::pair<size_t, size_t> SplitBitIndex(size_t index) {
    return {index >> 3, index & 7};
}

/**
 * Keeps only specified number of lower bits in the value.
 *
 * @param value  unsigned number
 * @param bits   number of bits to mask
 * @return value with masked lower bits
 */
template <typename T>
constexpr T LowerBits(T value, size_t bits) {
    static_assert (std::is_unsigned_v<T>, "type is not unsigned");
    return value & ~(std::numeric_limits<T>::max() << bits);
}

/**
 * Converts bit count to rounded up byte count.
 */
constexpr size_t ByteCount(size_t bitCount) noexcept {
    return (bitCount + 7) >> 3;
}

/**
 * Align up to byte boundary.
 */
constexpr size_t AlignToByte(size_t bitCount) noexcept {
    return (bitCount + 7) & ~size_t(7);
}

/**
 * Encode signed 32 bit integer as unsigned one.
 */
constexpr ui32 ZigZagEncode32(i32 n) noexcept {
    return (static_cast<ui32>(n) << 1) ^ static_cast<ui32>(n >> 31);
}

/**
 * Decode signed 32 bit integer from unsigned one.
 */
constexpr i32 ZigZagDecode32(ui32 n) noexcept {
    return static_cast<i32>((n >> 1) ^ (~(n & 1) + 1));
}

/**
 * Encode signed 64 bit integer as unsigned one.
 */
constexpr ui64 ZigZagEncode64(i64 n) noexcept {
    return (static_cast<ui64>(n) << 1) ^ static_cast<ui64>(n >> 63);
}

/**
 * Decode signed 64 bit integer from unsigned one.
 */
constexpr i64 ZigZagDecode64(ui64 n) noexcept {
    return static_cast<i64>((n >> 1) ^ (~(n & 1) + 1));
}

} // namespace NSolomon::NTs
