#pragma once

#include <util/generic/cast.h>
#include <util/generic/typetraits.h>
#include <util/system/types.h>

#include <array>
#include <numeric>
#include <ostream>

namespace NSolomon::NTs {

enum class EColumn: ui8 {
    TS            = 0,
    VALUE         = 1,
    MERGE         = 2,
    COUNT         = 3,
    STEP          = 4,
    RESERVED_DO_NOT_USE = 5, // vacant position
    HISTOGRAM     = 6,
    LOG_HISTOGRAM = 7,
    ISUMMARY      = 8,
    DSUMMARY      = 9,
    LONG_VALUE    = 10,
};

using TColumnsMask = ui16;

class TColumnSet {
    static constexpr TColumnsMask ToMask(EColumn column) noexcept {
        return static_cast<TColumnsMask>(1) << ToUnderlying(column);
    }

    template <typename TContainer>
    static constexpr TColumnsMask CombineMask(const TContainer& container) noexcept {
        TColumnsMask mask = 0;
        for (EColumn c: container) {
            if (c == EColumn::RESERVED_DO_NOT_USE) continue;
            mask |= ToMask(c);
        }
        return mask;
    }

public:
    static const std::array<EColumn, 11> AllColumns;
    static const TColumnsMask MaxMask;

public:
    constexpr TColumnSet() noexcept
        : Mask_{0}
    {
    }

    constexpr explicit TColumnSet(TColumnsMask mask) noexcept
        : Mask_{mask}
    {
    }

    constexpr explicit TColumnSet(EColumn column) noexcept
        : Mask_{ToMask(column)}
    {
    }

    constexpr TColumnSet(std::initializer_list<EColumn> columns) noexcept
        : Mask_{CombineMask(columns)}
    {
    }

    constexpr bool IsSet(EColumn c) const noexcept {
        return (Mask_ & ToMask(c)) != 0;
    }

    constexpr TColumnSet With(EColumn c) const noexcept {
        return *this | TColumnSet{c};
    }

    constexpr TColumnSet Without(EColumn c) const noexcept {
        return *this & ~TColumnSet{c};
    }

    constexpr auto Mask() const noexcept {
        return Mask_;
    }

    friend constexpr bool operator==(TColumnSet lhs, TColumnSet rhs) noexcept {
        return lhs.Mask_ == rhs.Mask_;
    }

    friend constexpr bool operator!=(TColumnSet lhs, TColumnSet rhs) noexcept {
        return !(lhs == rhs);
    }

    friend constexpr TColumnSet operator|(TColumnSet lhs, TColumnSet rhs) noexcept {
        return TColumnSet{static_cast<TColumnsMask>(lhs.Mask_ | rhs.Mask_)};
    }

    friend constexpr TColumnSet operator&(TColumnSet lhs, TColumnSet rhs) noexcept {
        return TColumnSet{static_cast<TColumnsMask>(lhs.Mask_ & rhs.Mask_)};
    }

    friend constexpr TColumnSet operator~(TColumnSet lhs) noexcept {
        return TColumnSet{static_cast<TColumnsMask>(~lhs.Mask_)};
    }

    friend std::ostream& operator<<(std::ostream& os, const TColumnSet& set);

private:
    TColumnsMask Mask_;
};

static_assert(sizeof(TColumnSet) == sizeof(TColumnsMask), "unexpected sizeof TColumnSet");

} // namespace NSolomon::NTs

Y_DECLARE_PODTYPE(NSolomon::NTs::TColumnSet);
