#pragma once

#include "base_ts_codec.h"

namespace NSolomon::NTs {

/**
 * Counter timeseries encoder.
 */
class TCounterTsEncoder: public TBaseTsEncoder<TCounterTsEncoder, NValue::TLong> {
    using TBase = TBaseTsEncoder<TCounterTsEncoder, NValue::TLong>;
    friend TBase;
public:
    TCounterTsEncoder(TColumnSet columns, TBitWriter* writer)
        : TBase{columns, writer}
    {
    }

    static TCounterTsEncoder Simple(TBitWriter* writer) {
        return TCounterTsEncoder{TLongPoint::SimpleColumns, writer};
    }

    static TCounterTsEncoder Aggr(TBitWriter* writer) {
        return TCounterTsEncoder{TLongPoint::AggrColumns, writer};
    }

private:
    void EncodeCommand(TBitWriter*, NValue::TLong) {
        // nop
    }

    void EncodeValue(TBitWriter* writer, NValue::TLong value) {
        long delta = value.Value - PrevValue_;
        long deltaOfDelta = delta - PrevValueDelta_;
        writer->WriteVarInt64Mode(ZigZagEncode64(deltaOfDelta));
        PrevValue_ = value.Value;
        PrevValueDelta_ = delta;
    }

    void WriteState(TBitWriter* writer) {
        writer->WriteVarInt64Mode(PrevValue_);
        PrevValue_ = 0;
        writer->WriteVarInt64Mode(PrevValueDelta_);
        PrevValueDelta_ = 0;
    }

private:
    i64 PrevValue_{0};
    i64 PrevValueDelta_{0};
};

/**
 * Counter timeseries decoder.
 */
class TCounterTsDecoder: public TBaseTsDecoder<TCounterTsDecoder, NValue::TLong> {
    using TBase = TBaseTsDecoder<TCounterTsDecoder, NValue::TLong>;
    friend TBase;
public:
    TCounterTsDecoder(TColumnSet columns, TBitSpan data)
        : TBase{columns, data}
    {
    }

    static TCounterTsDecoder Simple(TBitSpan data) {
        return TCounterTsDecoder{TLongPoint::SimpleColumns, data};
    }

    static TCounterTsDecoder Aggr(TBitSpan data) {
        return TCounterTsDecoder{TLongPoint::AggrColumns, data};
    }

private:
    void DecodeCommand(TBitReader*) {
        // nop
    }

    void DecodeValue(TBitReader* reader, NValue::TLong* value) {
        auto deltaOfDelta = reader->ReadVarInt64Mode();
        Y_ENSURE(deltaOfDelta.has_value(), "cannot read next delta of delta");

        i64 delta = PrevValueDelta_ + ZigZagDecode64(*deltaOfDelta);
        i64 v = PrevValue_ + delta;
        value->Value = v;
        PrevValue_ = v;
        PrevValueDelta_ = delta;
    }

    void Reset() {
        PrevValue_ = 0;
        PrevValueDelta_ = 0;
    }

private:
    i64 PrevValue_{0};
    i64 PrevValueDelta_{0};
};

} // namespace NSolomon::NTs
