#include "denom_codec.h"

namespace NSolomon::NTs {
namespace {

enum class EDenomType: ui8 {
    None,
    M1000,
    M1,
};

constexpr size_t TypeBits = 3;

} // namespace

ui64 DenomDecode(TBitReader* reader) {
    Y_ENSURE(reader->Left() >= TypeBits, "cannot read denom, left(" << reader->Left() << ") < " << TypeBits);
    auto type = static_cast<EDenomType>(reader->ReadInt8(3));

    switch (type) {
        case EDenomType::None:
            return 0;

        case EDenomType::M1000: {
            auto denom = reader->ReadVarInt64();
            Y_ENSURE(denom.has_value(), "cannot read denom M1000");
            return *denom * 1000;
        }

        case EDenomType::M1: {
            auto denom = reader->ReadVarInt64();
            Y_ENSURE(denom.has_value(), "cannot read denom M1");
            return *denom;
        }
    }

    ythrow yexception() << "unknown denom type: " << static_cast<ui8>(type);
}

void DenomEncode(TBitWriter* writer, ui64 denom) {
    if (denom == 0) {
        writer->WriteInt8(ToUnderlying(EDenomType::None), TypeBits);
    } else if (denom % 1000ull == 0) {
        writer->WriteInt8(ToUnderlying(EDenomType::M1000), TypeBits);
        writer->WriteVarInt64(denom / 1000ull);
    } else {
        writer->WriteInt8(ToUnderlying(EDenomType::M1), TypeBits);
        writer->WriteVarInt64(denom);
    }
}

} // namespace NSolomon::NTs
