#include "double_ts_codec.h"

#include <util/generic/bitops.h>

namespace NSolomon::NTs {

void TDoubleTsEncoder::EncodeValue(TBitWriter* writer, NValue::TDouble value) {
    ui64 xored = BitCast<ui64>(PrevValueNum_) ^ BitCast<ui64>(value.Num);
    if (xored == 0) {
        // 2. If XOR with the previous is zero (same value), store single ‘0’ bit
        writer->WriteBit(false);
        return;
    }

    ui8 lz = Min<ui8>(64 - GetValueBitCount(xored), 31);
    ui8 tz = CountTrailingZeroBits(xored);
    ui8 meaningfulBits = 0;

    // 3. When XOR is non-zero, calculate the number of leading
    // and trailing zeros in the XOR, store bit ‘1’ followed
    // by either a) or b):
    writer->WriteBit(true);

    if (lz >= PrevValueNumLeadingZeros_ && tz >= PrevValueNumTrailingZeros_ &&
        (PrevValueNumLeadingZeros_ != 0 || PrevValueNumTrailingZeros_ != 0))
    {
        // (a) (Control bit ‘0’) If the block of meaningful bits
        // falls within the block of previous meaningful bits,
        // i.e., there are at least as many leading zeros and
        // as many trailing zeros as with the previous value,
        // use that information for the block position and
        // just store the meaningful XORed value.

        tz = PrevValueNumTrailingZeros_;
        lz = PrevValueNumLeadingZeros_;
        meaningfulBits = 64 - PrevValueNumTrailingZeros_ - PrevValueNumLeadingZeros_;

        writer->WriteBit(false);
    } else {
        // (b) (Control bit ‘1’) Store the length of the number
        // of leading zeros in the next 5 bits, then store the
        // length of the meaningful XORed value in the next
        // 6 bits. Finally store the meaningful bits of the
        // XORed value.

        meaningfulBits = 64 - lz - tz;

        writer->WriteBit(true);
        writer->WriteInt8(lz, 5);
        writer->WriteInt8(meaningfulBits - 1, 6);
    }

    ui64 meaningfulXor = xored >> tz;
    writer->WriteInt64(meaningfulXor, meaningfulBits);

    PrevValueNum_ = value.Num;
    PrevValueNumTrailingZeros_ = tz;
    PrevValueNumLeadingZeros_ = lz;
}

void TDoubleTsDecoder::DecodeValue(TBitReader* reader, NValue::TDouble* value) {
    value->Denom = PrevValueDenom_;

    Y_ENSURE(reader->Left() > 0, "cannot read next double value, left 0 bits");
    if (!reader->ReadBit()) {
        value->Num = PrevValueNum_;
        return;
    }

    ui8 lz;
    ui8 tz;
    ui8 meaningfulBits;

    Y_ENSURE(reader->Left() > 0, "cannot read next double value, left 0 bits");
    if (!reader->ReadBit() && (PrevValueNumLeadingZeros_ != 0 || PrevValueNumTrailingZeros_ != 0)) {
        lz = PrevValueNumLeadingZeros_;
        tz = PrevValueNumTrailingZeros_;
        meaningfulBits = 64 - lz - tz;
    } else {
        Y_ENSURE(reader->Left() >= 11, "cannot read next double value, left(" << reader->Left() << ") < 11");
        lz = reader->ReadInt8(5);
        meaningfulBits = reader->ReadInt8(6) + 1;
        tz = 64 - meaningfulBits - lz;
    }

    Y_ENSURE(reader->Left() >= meaningfulBits,
            "cannot read next double value, left(" << reader->Left() << ") < " << static_cast<ui32>(meaningfulBits));
    ui64 xored = reader->ReadInt64(meaningfulBits);
    auto valueNum = BitCast<double>((xored << tz) ^ BitCast<ui64>(PrevValueNum_));

    PrevValueNum_ = valueNum;
    PrevValueNumLeadingZeros_ = lz;
    PrevValueNumTrailingZeros_ = tz;

    value->Num = valueNum;
}

} // namespace NSolomon::NTs
