#pragma once

#include "base_ts_codec.h"
#include "denom_codec.h"

namespace NSolomon::NTs {

/**
 * Double timeseries encoder.
 */
class TDoubleTsEncoder: public TBaseTsEncoder<TDoubleTsEncoder, NValue::TDouble> {
    using TBase = TBaseTsEncoder<TDoubleTsEncoder, NValue::TDouble>;
    friend TBase;
public:
    TDoubleTsEncoder(TColumnSet columns, TBitWriter* writer)
        : TBase{columns, writer}
    {
    }

    static TDoubleTsEncoder Simple(TBitWriter* writer) {
        return TDoubleTsEncoder{TDoublePoint::SimpleColumns, writer};
    }

    static TDoubleTsEncoder Aggr(TBitWriter* writer) {
        return TDoubleTsEncoder{TDoublePoint::AggrColumns, writer};
    }

private:
    void EncodeCommand(TBitWriter* writer, NValue::TDouble value) {
        if (Y_UNLIKELY(PrevValueDenom_ != value.Denom)) {
            writer->WriteBit(true);
            writer->WriteInt32(static_cast<ui32>(EColumn::VALUE), ColumnBits);
            DenomEncode(writer, value.Denom);
            PrevValueDenom_ = value.Denom;
        }
    }

    void EncodeValue(TBitWriter* writer, NValue::TDouble value);

    void WriteState(TBitWriter* writer) {
        DenomEncode(writer, PrevValueDenom_);
        PrevValueDenom_ = 0;

        writer->WriteDouble(PrevValueNum_);
        PrevValueNum_ = 0;

        writer->WriteInt8(PrevValueNumLeadingZeros_);
        PrevValueNumLeadingZeros_ = 0;

        writer->WriteInt8(PrevValueNumTrailingZeros_);
        PrevValueNumTrailingZeros_ = 0;
    }

private:
    double PrevValueNum_{0};
    ui64 PrevValueDenom_{0};
    ui8 PrevValueNumTrailingZeros_{0};
    ui8 PrevValueNumLeadingZeros_{0};
};

/**
 * Double timeseries decoder.
 */
class TDoubleTsDecoder: public TBaseTsDecoder<TDoubleTsDecoder, NValue::TDouble> {
    using TBase = TBaseTsDecoder<TDoubleTsDecoder, NValue::TDouble>;
    friend TBase;
public:
    TDoubleTsDecoder(TColumnSet columns, TBitSpan data)
        : TBase{columns, data}
    {
    }

    static TDoubleTsDecoder Simple(TBitSpan data) {
        return TDoubleTsDecoder{TDoublePoint::SimpleColumns, data};
    }

    static TDoubleTsDecoder Aggr(TBitSpan data) {
        return TDoubleTsDecoder{TDoublePoint::AggrColumns, data};
    }

private:
    void DecodeCommand(TBitReader* reader) {
        PrevValueDenom_ = DenomDecode(reader);
    }

    void DecodeValue(TBitReader* reader, NValue::TDouble* value);

    void Reset() {
        PrevValueNum_ = 0;
        PrevValueDenom_ = 0;
        PrevValueNumTrailingZeros_ = 0;
        PrevValueNumLeadingZeros_ = 0;
    }

private:
    double PrevValueNum_{0};
    ui64 PrevValueDenom_{0};
    ui8 PrevValueNumTrailingZeros_{0};
    ui8 PrevValueNumLeadingZeros_{0};
};

} // namespace NSolomon::NTs
