#pragma once

#include "bit_reader.h"
#include "bit_writer.h"

#include <util/generic/size_literals.h>

namespace NSolomon::NTs {

//
// Frame layout:
// ~~~~~~~~~~~~~
//
//   ________ header _________                                 ___________________ footer ______________________
//  /                         \                               /                                                 \
// +--------------------+------+============= / =============+---------------------+============= / =============+
// | frame payload size |  ,*  |           frame             | footer payload size |           footer            |
// |       (bytes)      | /    |          payload            |        (bytes)      |           payload           |
// +--------------------+/-----+============= / =============+---------------------+============= / =============+
//         ui32         /  ui8            var length                   ui32                 var length
//                     /           (aligned up to byte size)                         (aligned up to byte size)
//           used bits in the last
//             frame payload byte
//

// bit level sizes
constexpr size_t FRAME_HEADER_SIZE = BitsSize<ui32>() + BitsSize<ui8>();
constexpr size_t FRAME_FOOTER_SIZE = BitsSize<ui32>();
constexpr size_t FRAME_PAYLOAD_MIN_SIZE = 8 * 1_KB;

class TFrameEncoder {
public:
    void InitFrame(TBitWriter* writer) {
        FrameIdx_ = DoInitFrame(writer);
    }

    template <typename TFooterWriter>
    void FinishFrame(TBitWriter* writer, TFooterWriter footerWriter) {
        size_t usedBitsForLastByte = writer->Pos() & size_t(7);
        writer->AlignToByte();
        size_t frameFooterPos = writer->Pos();

        // update header
        {
            writer->SetPos(FrameIdx_);
            writer->WriteInt32(static_cast<ui32>((frameFooterPos - FrameIdx_ - FRAME_HEADER_SIZE) >> 3));
            writer->WriteInt8(usedBitsForLastByte);
        }

        // write footer
        {
            writer->SetPos(frameFooterPos);
            writer->WriteInt32(0); // size of state unknown right now
            footerWriter();
            writer->AlignToByte();

            size_t endPos = writer->Pos();
            writer->SetPos(frameFooterPos);
            writer->WriteInt32(static_cast<ui32>(endPos - frameFooterPos - FRAME_FOOTER_SIZE) >> 3);
            writer->SetPos(endPos);
        }
    }

    template <typename TFooterWriter>
    bool CloseFrame(TBitWriter* writer, TFooterWriter footerWriter) {
        if (FrameSize(writer) < FRAME_PAYLOAD_MIN_SIZE) {
            return false;
        }

        FinishFrame(writer, FrameIdx_, std::move(footerWriter));
        FrameIdx_ = DoInitFrame(writer);
        return true;
    }

    size_t FrameSize(TBitWriter* writer) {
        // align up to byte boundary
        return (writer->Pos() - FrameIdx_ + 7) & ~size_t(7);
    }

    size_t FrameIdx() const noexcept {
        return FrameIdx_;
    }

private:
    static size_t DoInitFrame(TBitWriter* writer) {
        size_t pos = writer->Pos();
        writer->WriteInt32(0);
        writer->WriteInt8(0);
        return pos;
    }

    static void ContinueFrame(TBitWriter* writer, size_t frameStartPos) {
        size_t pos = writer->Pos();
        writer->SetPos(frameStartPos);
        writer->WriteInt32(0);
        writer->WriteInt8(0);
        writer->SetPos(pos);
    }

private:
    size_t FrameIdx_{0};
};

class TFrameDecoder {
public:
    bool HasNext(TBitReader* reader) const noexcept {
        if (Y_UNLIKELY(HeaderIdx_ == -1)) {
            return reader->Left() > 0;
        }

        if (FooterIdx_ == -1) {
            return false;
        }

        return FooterIdx_ + FRAME_FOOTER_SIZE + FooterBits_ < reader->Size();
    }

    bool Next(TBitReader* reader);

    size_t PayloadIndex() const noexcept {
        Y_VERIFY_DEBUG(HeaderIdx_ >= 0);
        return static_cast<size_t>(HeaderIdx_) + FRAME_HEADER_SIZE;
    }

    size_t PayloadBits() const noexcept {
        Y_VERIFY_DEBUG(PayloadBits_ >= 0);
        return static_cast<size_t>(PayloadBits_);
    }

    bool HasFooter() const noexcept {
        return FooterIdx_ != -1;
    }

    size_t FooterIndex() const noexcept {
        Y_VERIFY_DEBUG(FooterIdx_ >= 0);
        return static_cast<size_t>(FooterIdx_) + FRAME_FOOTER_SIZE;
    }

    size_t FooterBits() const noexcept {
        Y_VERIFY_DEBUG(FooterBits_ >= 0);
        return static_cast<size_t>(FooterBits_);
    }

private:
    ssize_t HeaderIdx_{-1};
    ssize_t PayloadBits_{-1};
    ssize_t FooterIdx_{-1};
    ssize_t FooterBits_{-1};
};

} // namespace NSolomon::NTs
