#pragma once

#include "base_ts_codec.h"

namespace NSolomon::NTs {

/**
 * Integer gauge timeseries encoder.
 */
class TGaugeIntTsEncoder: public TBaseTsEncoder<TGaugeIntTsEncoder, NValue::TLong> {
    using TBase = TBaseTsEncoder<TGaugeIntTsEncoder, NValue::TLong>;
    friend TBase;
public:
    TGaugeIntTsEncoder(TColumnSet columns, TBitWriter* writer)
        : TBase{columns, writer}
    {
    }

    static TGaugeIntTsEncoder Simple(TBitWriter* writer) {
        return TGaugeIntTsEncoder{TLongPoint::SimpleColumns, writer};
    }

    static TGaugeIntTsEncoder Aggr(TBitWriter* writer) {
        return TGaugeIntTsEncoder{TLongPoint::AggrColumns, writer};
    }

private:
    void EncodeCommand(TBitWriter*, NValue::TLong) {
        // nop
    }

    void EncodeValue(TBitWriter* writer, NValue::TLong value) {
        i64 delta = value.Value - PrevValue_;
        writer->WriteVarInt64Mode(ZigZagEncode64(delta));
        PrevValue_ = value.Value;
    }

    void WriteState(TBitWriter* writer) {
        writer->WriteVarInt64Mode(PrevValue_);
        PrevValue_ = 0;
    }

private:
    i64 PrevValue_{0};
};

/**
 * Integer gauge timeseries decoder.
 */
class TGaugeIntTsDecoder: public TBaseTsDecoder<TGaugeIntTsDecoder, NValue::TLong> {
    using TBase = TBaseTsDecoder<TGaugeIntTsDecoder, NValue::TLong>;
    friend TBase;
public:
    TGaugeIntTsDecoder(TColumnSet columns, TBitSpan data)
        : TBase{columns, data}
    {
    }

    static TGaugeIntTsDecoder Simple(TBitSpan data) {
        return TGaugeIntTsDecoder{TLongPoint::SimpleColumns, data};
    }

    static TGaugeIntTsDecoder Aggr(TBitSpan data) {
        return TGaugeIntTsDecoder{TLongPoint::AggrColumns, data};
    }

private:
    void DecodeCommand(TBitReader*) {
        // nop
    }

    void DecodeValue(TBitReader* reader, NValue::TLong* value) {
        auto delta = reader->ReadVarInt64Mode();
        Y_ENSURE(delta.has_value(), "cannot read next delta of delta");

        i64 v = PrevValue_ + ZigZagDecode64(*delta);
        PrevValue_ = value->Value = v;
    }

    void Reset() {
        PrevValue_ = 0;
    }

private:
    i64 PrevValue_{0};
};

} // namespace NSolomon::NTs
