#include "gorilla_codec.h"

namespace NSolomon::NTs {

void TGorillaEncoder::Encode(TBitWriter *writer, double value) {
    ui64 xored = BitCast<ui64>(Prev_) ^ BitCast<ui64>(value);
    if (xored == 0) {
        // 2. If XOR with the previous is zero (same value), store single ‘0’ bit
        writer->WriteBit(false);
        return;
    }

    ui8 lz = Min<ui8>(64 - GetValueBitCount(xored), 31);
    ui8 tz = CountTrailingZeroBits(xored);
    ui8 meaningfulBits = 0;

    // 3. When XOR is non-zero, calculate the number of leading
    // and trailing zeros in the XOR, store bit ‘1’ followed
    // by either a) or b):
    writer->WriteBit(true);

    if (lz >= PrevLeadingZeros_ && tz >= PrevTrailingZeros_ &&
        (PrevLeadingZeros_ != 0 || PrevTrailingZeros_ != 0))
    {
        // (a) (Control bit ‘0’) If the block of meaningful bits
        // falls within the block of previous meaningful bits,
        // i.e., there are at least as many leading zeros and
        // as many trailing zeros as with the previous value,
        // use that information for the block position and
        // just store the meaningful XORed value.

        tz = PrevTrailingZeros_;
        lz = PrevLeadingZeros_;
        meaningfulBits = 64 - PrevTrailingZeros_ - PrevLeadingZeros_;

        writer->WriteBit(false);
    } else {
        // (b) (Control bit ‘1’) Store the length of the number
        // of leading zeros in the next 5 bits, then store the
        // length of the meaningful XORed value in the next
        // 6 bits. Finally store the meaningful bits of the
        // XORed value.

        meaningfulBits = 64 - lz - tz;

        writer->WriteBit(true);
        writer->WriteInt8(lz, 5);
        writer->WriteInt8(meaningfulBits - 1, 6);
    }

    ui64 meaningfulXor = xored >> tz;
    writer->WriteInt64(meaningfulXor, meaningfulBits);

    Prev_ = value;
    PrevTrailingZeros_ = tz;
    PrevLeadingZeros_ = lz;
}

double TGorillaDecoder::Decode(TBitReader *reader) {
    Y_ENSURE(reader->Left() > 0, "cannot read next double value, left 0 bits");
    if (!reader->ReadBit()) {
        return Prev_;
    }

    ui8 lz;
    ui8 tz;
    ui8 meaningfulBits;

    Y_ENSURE(reader->Left() > 0, "cannot read next double value, left 0 bits");
    if (!reader->ReadBit() && (PrevLeadingZeros_ != 0 || PrevTrailingZeros_ != 0)) {
        lz = PrevLeadingZeros_;
        tz = PrevTrailingZeros_;
        meaningfulBits = 64 - lz - tz;
    } else {
        Y_ENSURE(reader->Left() >= 11, "cannot read next double value, left(" << reader->Left() << ") < 11");
        lz = reader->ReadInt8(5);
        meaningfulBits = reader->ReadInt8(6) + 1;
        tz = 64 - meaningfulBits - lz;
    }

    Y_ENSURE(reader->Left() >= meaningfulBits,
             "cannot read next double value, left(" << reader->Left() << ") < " << static_cast<ui32>(meaningfulBits));
    ui64 xored = reader->ReadInt64(meaningfulBits);
    auto value = BitCast<double>((xored << tz) ^ BitCast<ui64>(Prev_));

    Prev_ = value;
    PrevLeadingZeros_ = lz;
    PrevTrailingZeros_ = tz;

    return value;
}

} // namespace NSolomon::NTs
