#pragma once

#include "base_ts_codec.h"
#include "gorilla_codec.h"

namespace NSolomon::NTs {

inline constexpr ui8 MaxBucketCountBits = 7;

/**
 * Log histogram timeseries encoder.
 */
class TLogHistogramTsEncoder: public TBaseTsEncoder<TLogHistogramTsEncoder, NValue::TLogHistogram> {
    using TBase = TBaseTsEncoder<TLogHistogramTsEncoder, NValue::TLogHistogram>;
    friend TBase;
public:
    TLogHistogramTsEncoder(TColumnSet columns, TBitWriter* writer)
        : TBase{columns, writer}
    {
    }

    static TLogHistogramTsEncoder Simple(TBitWriter* writer) {
        return TLogHistogramTsEncoder{TLogHistogramPoint::SimpleColumns, writer};
    }

    static TLogHistogramTsEncoder Aggr(TBitWriter* writer) {
        return TLogHistogramTsEncoder{TLogHistogramPoint::AggrColumns, writer};
    }

private:
    void EncodeCommand(TBitWriter* writer, const NValue::TLogHistogram& value) {
        if (PrevMaxBucketCount_ != value.MaxBucketCount || PrevBase_ != value.Base) {
            writer->WriteBit(true);
            writer->WriteInt32(static_cast<ui32>(EColumn::LOG_HISTOGRAM), ColumnBits);

            PrevMaxBucketCount_ = value.MaxBucketCount;
            writer->WriteInt8(PrevMaxBucketCount_, MaxBucketCountBits);

            PrevBase_ = value.Base;
            writer->WriteDouble(PrevBase_);
        }
    }

    void EncodeValue(TBitWriter* writer, const NValue::TLogHistogram& hist) {
        writer->WriteVarInt32Mode(ZigZagEncode32(hist.StartPower - PrevStartPower_));
        PrevStartPower_ = hist.StartPower;

        writer->WriteVarInt64Mode(ZigZagEncode64(static_cast<i64>(hist.ZeroCount) - static_cast<i64>(PrevZeroCount_)));
        PrevZeroCount_ = hist.ZeroCount;

        writer->WriteVarInt32Mode(ZigZagEncode32(static_cast<i8>(hist.Values.size()) - static_cast<i8>(Size_)));
        Size_ = hist.Values.size();

        if (ValueEncoders_.size() < Size_) {
            ValueEncoders_.resize(Size_);
            // XXX: on shrinking down we have to keep decoders state
            //      for compatibility with java version
        }

        for (ui8 i = 0; i < Size_; ++i) {
            ValueEncoders_[i].Encode(writer, hist.Values[i]);
        }
    }

    void WriteState(TBitWriter* writer) {
        writer->WriteVarInt32Mode(PrevStartPower_);
        PrevStartPower_ = 0;

        writer->WriteVarInt64Mode(PrevZeroCount_);
        PrevZeroCount_ = 0;

        writer->WriteVarInt32Mode(Size_);
        PrevStartPower_ = 0;

        writer->WriteVarInt32Mode(PrevMaxBucketCount_);
        PrevMaxBucketCount_ = NValue::TLogHistogram::DefaultMaxBucketCount;

        writer->WriteDouble(PrevBase_);
        PrevBase_ = NValue::TLogHistogram::DefaultBase;

        writer->WriteVarInt32Mode(ValueEncoders_.size());
        for (auto& e: ValueEncoders_) {
            e.WriteState(writer);
        }
    }

private:
    std::vector<TGorillaEncoder> ValueEncoders_;
    ui64 PrevZeroCount_{0};
    double PrevBase_{NValue::TLogHistogram::DefaultBase};
    i16 PrevStartPower_{0};
    i16 PrevMaxBucketCount_{NValue::TLogHistogram::DefaultMaxBucketCount};
    ui8 Size_{0};
};

/**
 * Histogram timeseries decoder.
 */
class TLogHistogramTsDecoder: public TBaseTsDecoder<TLogHistogramTsDecoder, NValue::TLogHistogram> {
    using TBase = TBaseTsDecoder<TLogHistogramTsDecoder, NValue::TLogHistogram>;
    friend TBase;
public:
    TLogHistogramTsDecoder(TColumnSet columns, TBitSpan data)
        : TBase{columns, data}
    {
    }

    static TLogHistogramTsDecoder Simple(TBitSpan data) {
        return TLogHistogramTsDecoder{TLogHistogramPoint::SimpleColumns, data};
    }

    static TLogHistogramTsDecoder Aggr(TBitSpan data) {
        return TLogHistogramTsDecoder{TLogHistogramPoint::AggrColumns, data};
    }

private:
    void DecodeCommand(TBitReader* reader) {
        PrevMaxBucketCount_ = reader->ReadInt8(MaxBucketCountBits);
        PrevBase_ = reader->ReadDouble();
    }

    void DecodeValue(TBitReader* reader, NValue::TLogHistogram* hist) {
        auto startPower = reader->ReadVarInt32Mode();
        Y_ENSURE(startPower.has_value(), "cannot read log histogram start power");
        PrevStartPower_ += ZigZagDecode32(*startPower);

        auto zeroCount = reader->ReadVarInt64Mode();
        Y_ENSURE(zeroCount.has_value(), "cannot read log histogram zero count");
        PrevZeroCount_ += ZigZagDecode64(*zeroCount);

        auto size = reader->ReadVarInt32Mode();
        Y_ENSURE(size.has_value(), "cannot read log histogram zero count");
        Size_ += ZigZagDecode32(*size);

        hist->MaxBucketCount = PrevMaxBucketCount_;
        hist->Base = PrevBase_;
        hist->StartPower = PrevStartPower_;
        hist->ZeroCount = PrevZeroCount_;

        if (ValueDecoders_.size() < Size_) {
            ValueDecoders_.resize(Size_);
            // XXX: on shrinking down we have to keep decoders state
            //      for compatibility with java version
        }

        hist->Values.resize(Size_);
        for (ui8 i = 0; i < Size_; ++i) {
            hist->Values[i] = ValueDecoders_[i].Decode(reader);
        }
    }

    void Reset() {
        ValueDecoders_.clear();
        PrevZeroCount_ = 0;
        PrevBase_ = NValue::TLogHistogram::DefaultBase;
        PrevStartPower_ = 0;
        PrevMaxBucketCount_ = NValue::TLogHistogram::DefaultMaxBucketCount;
        Size_ = 0;
    }

private:
    std::vector<TGorillaDecoder> ValueDecoders_;
    ui64 PrevZeroCount_{0};
    double PrevBase_{NValue::TLogHistogram::DefaultBase};
    i16 PrevStartPower_{0};
    i16 PrevMaxBucketCount_{NValue::TLogHistogram::DefaultMaxBucketCount};
    ui8 Size_{0};
};

} // namespace NSolomon::NTs
