#pragma once

#include "base_ts_codec.h"
#include "denom_codec.h"
#include "gorilla_codec.h"

namespace NSolomon::NTs {

/**
 * Histogram timeseries encoder.
 */
class THistogramTsEncoder: public TBaseTsEncoder<THistogramTsEncoder, NValue::THistogram> {
    using TBase = TBaseTsEncoder<THistogramTsEncoder, NValue::THistogram>;
    friend TBase;
public:
    THistogramTsEncoder(TColumnSet columns, TBitWriter* writer)
        : TBase{columns, writer}
    {
    }

    static THistogramTsEncoder Simple(TBitWriter* writer) {
        return THistogramTsEncoder{THistogramPoint::SimpleColumns, writer};
    }

    static THistogramTsEncoder Aggr(TBitWriter* writer) {
        return THistogramTsEncoder{THistogramPoint::AggrColumns, writer};
    }

private:
    void EncodeCommand(TBitWriter* writer, const NValue::THistogram& value) {
        Y_VERIFY_DEBUG(value.Buckets.size() <= std::numeric_limits<decltype(Size_)>::max());

        if (PrevDenom_ != value.Denom || !BoundsEqual(value)){
            writer->WriteBit(true);
            writer->WriteInt32(static_cast<ui32>(EColumn::HISTOGRAM), ColumnBits);

            size_t newSize = value.Buckets.size();
            if (Size_ < newSize) {
                BoundEncoders_.resize(newSize);
                PrevValues_.resize(newSize);
            }

            Size_ = newSize;
            writer->WriteVarInt32(Size_);

            for (size_t i = 0; i < newSize; ++i) {
                BoundEncoders_[i].Encode(writer, value.Buckets[i].UpperBound);
            }

            PrevDenom_ = value.Denom;
            DenomEncode(writer, PrevDenom_);
        }
    }

    void EncodeValue(TBitWriter* writer, const NValue::THistogram& value) {
        Y_VERIFY_DEBUG(value.Buckets.size() <= PrevValues_.size());

        if (value.Buckets.empty()) {
            return;
        }

        for (size_t i = 0; i < value.Buckets.size(); ++i) {
            i64 delta = static_cast<i64>(value.Buckets[i].Value) - static_cast<i64>(PrevValues_[i]);
            writer->WriteVarInt64Mode(ZigZagEncode64(delta));
            PrevValues_[i] = value.Buckets[i].Value;
        }
    }

    void WriteState(TBitWriter* writer) {
        // TODO: it is better to encode size as varint32
        writer->WriteVarInt32Mode(Size_);

        for (size_t i = 0; i < BoundEncoders_.size(); ++i) {
            BoundEncoders_[i].WriteState(writer);

            // TODO: it is better to write it after ZigZag encoding
            writer->WriteVarInt64Mode(PrevValues_[i]);
            PrevValues_[i] = 0;
        }

        // TODO: remove second write of Size_ (kept for compatibility with Java version)
        writer->WriteVarInt32Mode(Size_);
        Size_ = 0;

        DenomEncode(writer, PrevDenom_);
        PrevDenom_ = 0;
    }

    bool BoundsEqual(const NValue::THistogram& value) const noexcept {
        Y_VERIFY_DEBUG(Size_ <= BoundEncoders_.size());

        if (Size_ != value.Buckets.size()) {
            return false;
        }

        for (ui8 i = 0; i < Size_; ++i) {
            if (value.Buckets[i].UpperBound != BoundEncoders_[i].Prev()) {
                return false;
            }
        }

        return true;
    }

private:
    std::vector<TGorillaEncoder> BoundEncoders_;
    std::vector<ui64> PrevValues_;
    ui64 PrevDenom_{0};
    ui8 Size_{0};
};

/**
 * Histogram timeseries decoder.
 */
class THistogramTsDecoder: public TBaseTsDecoder<THistogramTsDecoder, NValue::THistogram> {
    using TBase = TBaseTsDecoder<THistogramTsDecoder, NValue::THistogram>;
    friend TBase;
public:
    THistogramTsDecoder(TColumnSet columns, TBitSpan data)
        : TBase{columns, data}
    {
    }

    static THistogramTsDecoder Simple(TBitSpan data) {
        return THistogramTsDecoder{THistogramPoint::SimpleColumns, data};
    }

    static THistogramTsDecoder Aggr(TBitSpan data) {
        return THistogramTsDecoder{THistogramPoint::AggrColumns, data};
    }

private:
    void DecodeCommand(TBitReader* reader) {
        auto size = reader->ReadVarInt32();
        Y_ENSURE(size.has_value(), "cannot read histogram buckets size");

        if (*size > Size_) {
            BoundDecoders_.resize(*size);
            PrevValues_.resize(*size);
            // XXX: on shrinking down we have to keep decoders state
            //      for compatibility with java version
        }

        Size_ = *size;
        for (ui8 i = 0; i < Size_; ++i) {
            BoundDecoders_[i].Decode(reader);
        }

        PrevDenom_ = DenomDecode(reader);
    }

    void DecodeValue(TBitReader* reader, NValue::THistogram* value) {
        value->Denom = PrevDenom_;
        value->Buckets.resize(Size_);

        for (ui8 i = 0; i < Size_; ++i) {
            auto delta = reader->ReadVarInt64Mode();
            Y_ENSURE(delta.has_value(), "cannot read " << static_cast<ui32>(i) << " bucket value");
            PrevValues_[i] += ZigZagDecode64(*delta);

            auto& bucket = value->Buckets[i];
            bucket.Value = PrevValues_[i];
            bucket.UpperBound = BoundDecoders_[i].Prev();
        }
    }

    void Reset() {
        BoundDecoders_.clear();
        PrevValues_.clear();
        PrevDenom_ = 0;
        Size_ = 0;
    }

private:
    std::vector<TGorillaDecoder> BoundDecoders_;
    std::vector<ui64> PrevValues_;
    ui64 PrevDenom_{0};
    ui8 Size_{0};
};

} // namespace NSolomon::NTs
