#pragma once

#include "columns.h"

#include <util/datetime/base.h>

#include <variant>
#include <vector>

namespace NSolomon::NTs {
namespace NValue {

struct TDouble {
    double Num{0.0};
    ui64 Denom{0};

    double ValueDivided() const noexcept {
        return Denom == 0 || Denom == 1000
            ? static_cast<double>(Num)
            : static_cast<double>(Num * 1000) / static_cast<double>(Denom);
    }

    bool operator==(const TDouble&) const = default;
};

struct TLong {
    i64 Value{0};

    bool operator==(const TLong&) const = default;
};

template <typename T>
struct TSummary {
    i64 CountValue{0};
    T Sum{};
    T Min{};
    T Max{};
    T Last{};

    bool operator==(const TSummary<T>&) const = default;
};

using TSummaryInt = TSummary<i64>;
using TSummaryDouble = TSummary<double>;

struct THistogram {
    struct TBucket {
        double UpperBound{0.0};
        ui64 Value{0};

        bool operator==(const TBucket&) const = default;
    };

    ui64 Denom{0};
    std::vector<TBucket> Buckets;

    bool operator==(const THistogram&) const = default;
};

struct TLogHistogram {
    inline static constexpr auto DefaultMaxBucketCount = 100;
    inline static constexpr double DefaultBase = 1.5;

    std::vector<double> Values; // values stored in buckets
    ui64 ZeroCount{0};
    i16 StartPower{0};
    i16 MaxBucketCount{DefaultMaxBucketCount};
    double Base{DefaultBase};

    bool operator==(const TLogHistogram&) const = default;
};

} // namespace NValue

/**
 * Allows to get value from a point. Template implementation just casts
 * point to a value type, because all point types are inherit value types.
 */
template <typename TPoint, typename TValue>
struct TValueGetter {
    static constexpr const TValue& Get(const TPoint& point) {
        return static_cast<const TValue&>(point);
    }

    static constexpr TValue& Get(TPoint& point) {
        return static_cast<TValue&>(point);
    }

    static constexpr TValue* Mut(TPoint* point) {
        return static_cast<TValue*>(point);
    }
};

struct TPointCommon {
    TInstant Time;
    TDuration Step;
    ui64 Count{0};
    bool Merge{false};
};

/**
 * Point with double value.
 */
struct TDoublePoint: public TPointCommon, public NValue::TDouble {
    using TValue = NValue::TDouble;
    static constexpr TColumnSet MinColumns = { EColumn::TS, EColumn::VALUE };
    static constexpr TColumnSet SimpleColumns = { EColumn::TS, EColumn::VALUE, EColumn::STEP };
    static constexpr TColumnSet AggrColumns = SimpleColumns | TColumnSet{ EColumn::MERGE, EColumn::COUNT };
};

/**
 * Point with long value.
 */
struct TLongPoint: public TPointCommon, public NValue::TLong {
    using TValue = NValue::TLong;
    static constexpr TColumnSet MinColumns = { EColumn::TS, EColumn::LONG_VALUE };
    static constexpr TColumnSet SimpleColumns = { EColumn::TS, EColumn::LONG_VALUE, EColumn::STEP };
    static constexpr TColumnSet AggrColumns = SimpleColumns | TColumnSet{ EColumn::MERGE, EColumn::COUNT };
};

/**
 * Point with integer summary value.
 */
struct TSummaryIntPoint: public TPointCommon, public NValue::TSummaryInt {
    using TValue = NValue::TSummaryInt;
    static constexpr TColumnSet MinColumns = { EColumn::TS, EColumn::ISUMMARY };
    static constexpr TColumnSet SimpleColumns = { EColumn::TS, EColumn::ISUMMARY, EColumn::STEP };
    static constexpr TColumnSet AggrColumns = SimpleColumns | TColumnSet{ EColumn::MERGE, EColumn::COUNT };
    static constexpr auto EmptyValue = NValue::TSummaryInt{
            .CountValue = 0,
            .Sum = 0,
            .Min = std::numeric_limits<i64>::max(),
            .Max = std::numeric_limits<i64>::min(),
            .Last = 0
    };
};

/**
 * Point with double summary value.
 */
struct TSummaryDoublePoint: public TPointCommon, public NValue::TSummaryDouble {
    using TValue = NValue::TSummaryDouble;
    static constexpr TColumnSet MinColumns = { EColumn::TS, EColumn::DSUMMARY };
    static constexpr TColumnSet SimpleColumns = { EColumn::TS, EColumn::DSUMMARY, EColumn::STEP };
    static constexpr TColumnSet AggrColumns = SimpleColumns | TColumnSet{ EColumn::MERGE, EColumn::COUNT };
    static constexpr auto EmptyValue = NValue::TSummaryDouble{
            .CountValue = 0,
            .Sum = 0.0,
            .Min = std::numeric_limits<double>::infinity(),
            .Max = -std::numeric_limits<double>::infinity(),
            .Last = 0.0
    };
};

/**
 * Point with histogram value.
 */
struct THistogramPoint: public TPointCommon, public NValue::THistogram {
    using TValue = NValue::THistogram;
    static constexpr TColumnSet MinColumns = { EColumn::TS, EColumn::HISTOGRAM };
    static constexpr TColumnSet SimpleColumns = { EColumn::TS, EColumn::HISTOGRAM, EColumn::STEP };
    static constexpr TColumnSet AggrColumns = SimpleColumns | TColumnSet{ EColumn::MERGE, EColumn::COUNT };
};

/**
 * Point with log-histogram value.
 */
struct TLogHistogramPoint: public TPointCommon, public NValue::TLogHistogram {
    using TValue = NValue::TLogHistogram;
    static constexpr TColumnSet MinColumns = { EColumn::TS, EColumn::LOG_HISTOGRAM };
    static constexpr TColumnSet SimpleColumns = { EColumn::TS, EColumn::LOG_HISTOGRAM, EColumn::STEP };
    static constexpr TColumnSet AggrColumns = SimpleColumns | TColumnSet{ EColumn::MERGE, EColumn::COUNT };
};

/**
 * Point that can hold any type of value.
 */
struct TVariantPoint: public TPointCommon {
    using TValue = std::variant<
        NValue::TDouble,
        NValue::TLong,
        NValue::TSummaryInt,
        NValue::TSummaryDouble,
        NValue::THistogram,
        NValue::TLogHistogram>;

    TValue Value;

    template <typename TValue>
    const TValue& Get() const {
        return std::get<TValue>(Value);
    }

    template <typename TValue>
    TValue& Get() {
        return std::get<TValue>(Value);
    }

    template <typename TValue>
    TValue* Mut() {
        if (Y_LIKELY(std::holds_alternative<TValue>(Value))) {
            return &std::get<TValue>(Value);
        }
        return &Value.emplace<TValue>();
    }

    template <typename TValue>
    void Set(const TValue& value) {
        Value = value;
    }

    template <typename TValue>
    void Set(TValue&& value) {
        Value = std::forward<TValue>(value);
    }

    void Set(const TVariantPoint& rhs) {
        Value = rhs.Value;
    }

    void Set(TVariantPoint&& point) {
        Value = std::move(point.Value);
    }
};

template <typename TValue>
struct TValueGetter<TVariantPoint, TValue> {
    static constexpr const TValue& Get(const TVariantPoint& point) {
        return point.Get<TValue>();
    }

    static constexpr TValue& Get(TVariantPoint& point) {
        return point.Get<TValue>();
    }

    static constexpr TValue* Mut(TVariantPoint* point) {
        return point->Mut<TValue>();
    }
};

} // namespace NSolomon::NTs
