#pragma once

#include "base_ts_codec.h"
#include "gorilla_codec.h"
#include "summary_mask.h"

namespace NSolomon::NTs {

/**
 * Double summary timeseries encoder.
 */
class TSummaryDoubleTsEncoder: public TBaseTsEncoder<TSummaryDoubleTsEncoder, NValue::TSummaryDouble> {
    using TBase = TBaseTsEncoder<TSummaryDoubleTsEncoder, NValue::TSummaryDouble>;
    friend TBase;
    static constexpr auto EmptyValue = TSummaryDoublePoint::EmptyValue;
public:
    TSummaryDoubleTsEncoder(TColumnSet columns, TBitWriter* writer)
        : TBase{columns, writer}
    {
    }

    static TSummaryDoubleTsEncoder Simple(TBitWriter* writer) {
        return TSummaryDoubleTsEncoder{TSummaryDoublePoint::SimpleColumns, writer};
    }

    static TSummaryDoubleTsEncoder Aggr(TBitWriter* writer) {
        return TSummaryDoubleTsEncoder{TSummaryDoublePoint::AggrColumns, writer};
    }

private:
    void EncodeCommand(TBitWriter* writer, const NValue::TSummaryDouble& value) {
        TSummaryMask mask;
        if (value.CountValue != EmptyValue.CountValue) {
            mask.SetCount();
        }

        if (value.Sum != EmptyValue.Sum) {
            mask.SetSum();
        }

        if (value.Min != EmptyValue.Min) {
            mask.SetMin();
        }

        if (value.Max != EmptyValue.Max) {
            mask.SetMax();
        }

        if (value.Last != EmptyValue.Last) {
            mask.SetLast();
        }

        if ((Mask_ & mask) != mask) {
            Mask_ = Mask_ | mask;
            writer->WriteBit(true);
            writer->WriteInt8(static_cast<ui8>(EColumn::ISUMMARY), ColumnBits);
            writer->WriteInt8(mask, 5);
        }
    }

    void EncodeValue(TBitWriter* writer, const NValue::TSummaryDouble& value) {
        if (Mask_.HasCount()) {
            i64 delta = value.CountValue - PrevCount_;
            writer->WriteVarInt64Mode(ZigZagEncode64(delta));
            PrevCount_ = value.CountValue;
        }

        if (Mask_.HasSum()) {
            SumEncoder_.Encode(writer, value.Sum);
        }

        if (Mask_.HasMin()) {
            MinEncoder_.Encode(writer, value.Min);
        }

        if (Mask_.HasMax()) {
            MaxEncoder_.Encode(writer, value.Max);
        }

        if (Mask_.HasLast()) {
            LastEncoder_.Encode(writer, value.Last);
        }
    }

    void WriteState(TBitWriter* writer) {
        writer->WriteInt8(Mask_, 5);
        Mask_ = {};

        // XXX: must be zig-zag encoded, but kept for backward compatibility with Java version
        writer->WriteVarInt64Mode(static_cast<ui64>(PrevCount_));
        PrevCount_ = 0;

        SumEncoder_.WriteState(writer);
        MinEncoder_.WriteState(writer);
        MaxEncoder_.WriteState(writer);
        LastEncoder_.WriteState(writer);
    }

private:
    i64 PrevCount_{0};
    TGorillaEncoder SumEncoder_;
    TGorillaEncoder MinEncoder_;
    TGorillaEncoder MaxEncoder_;
    TGorillaEncoder LastEncoder_;
    TSummaryMask Mask_;
};

/**
 * Double summary timeseries decoder.
 */
class TSummaryDoubleTsDecoder: public TBaseTsDecoder<TSummaryDoubleTsDecoder, NValue::TSummaryDouble> {
    using TBase = TBaseTsDecoder<TSummaryDoubleTsDecoder, NValue::TSummaryDouble>;
    friend TBase;
    static constexpr auto EmptyValue = TSummaryDoublePoint::EmptyValue;
public:
    TSummaryDoubleTsDecoder(TColumnSet columns, TBitSpan data)
        : TBase{columns, data}
    {
    }

    static TSummaryDoubleTsDecoder Simple(TBitSpan data) {
        return TSummaryDoubleTsDecoder{TSummaryDoublePoint::SimpleColumns, data};
    }

    static TSummaryDoubleTsDecoder Aggr(TBitSpan data) {
        return TSummaryDoubleTsDecoder{TSummaryDoublePoint::AggrColumns, data};
    }

private:
    void DecodeCommand(TBitReader* reader) {
        Mask_ = TSummaryMask{reader->ReadInt8(5)};
    }

    void DecodeValue(TBitReader* reader, NValue::TSummaryDouble* value) {
        if (Mask_.HasCount()) {
            auto value = reader->ReadVarInt64Mode();
            Y_ENSURE(value.has_value(), "cannot read DSUMMARY count field");
            PrevCount_ += ZigZagDecode64(*value);
        }

        value->CountValue = PrevCount_;
        value->Sum = Mask_.HasSum() ? SumDecoder_.Decode(reader) : EmptyValue.Sum;
        value->Min = Mask_.HasMin() ? MinDecoder_.Decode(reader) : EmptyValue.Min;
        value->Max = Mask_.HasMax() ? MaxDecoder_.Decode(reader) : EmptyValue.Max;
        value->Last = Mask_.HasLast() ? LastDecoder_.Decode(reader) : EmptyValue.Last;
    }

    void Reset() {
        Mask_ = {};
        PrevCount_ = 0;
        SumDecoder_ = {};
        MinDecoder_ = {};
        MaxDecoder_ = {};
        LastDecoder_ = {};
    }

private:
    i64 PrevCount_{0};
    TGorillaDecoder SumDecoder_;
    TGorillaDecoder MinDecoder_;
    TGorillaDecoder MaxDecoder_;
    TGorillaDecoder LastDecoder_;
    TSummaryMask Mask_;
};

} // namespace NSolomon::NTs
