#pragma once

#include "base_ts_codec.h"
#include "summary_mask.h"

namespace NSolomon::NTs {

/**
 * Integer summary timeseries encoder.
 */
class TSummaryIntTsEncoder: public TBaseTsEncoder<TSummaryIntTsEncoder, NValue::TSummaryInt> {
    using TBase = TBaseTsEncoder<TSummaryIntTsEncoder, NValue::TSummaryInt>;
    friend TBase;
    static constexpr auto EmptyValue = TSummaryIntPoint::EmptyValue;
public:
    TSummaryIntTsEncoder(TColumnSet columns, TBitWriter* writer)
        : TBase{columns, writer}
    {
    }

    static TSummaryIntTsEncoder Simple(TBitWriter* writer) {
        return TSummaryIntTsEncoder{TSummaryIntPoint::SimpleColumns, writer};
    }

    static TSummaryIntTsEncoder Aggr(TBitWriter* writer) {
        return TSummaryIntTsEncoder{TSummaryIntPoint::AggrColumns, writer};
    }

private:
    void EncodeCommand(TBitWriter* writer, const NValue::TSummaryInt& value) {
        TSummaryMask mask;
        if (value.CountValue != EmptyValue.CountValue) {
            mask.SetCount();
        }

        if (value.Sum != EmptyValue.Sum) {
            mask.SetSum();
        }

        if (value.Min != EmptyValue.Min) {
            mask.SetMin();
        }

        if (value.Max != EmptyValue.Max) {
            mask.SetMax();
        }

        if (value.Last != EmptyValue.Last) {
            mask.SetLast();
        }

        if ((Mask_ & mask) != mask) {
            Mask_ = Mask_ | mask;
            writer->WriteBit(true);
            writer->WriteInt8(static_cast<ui8>(EColumn::ISUMMARY), ColumnBits);
            writer->WriteInt8(mask, 5);
        }
    }

    void EncodeValue(TBitWriter* writer, const NValue::TSummaryInt& value) {
        auto writeDelta = [writer](i64 current, i64 prev) {
            i64 delta = current - prev;
            writer->WriteVarInt64Mode(ZigZagEncode64(delta));
            return current;
        };

        if (Mask_.HasCount()) {
            PrevCount_ = writeDelta(value.CountValue, PrevCount_);
        }

        if (Mask_.HasSum()) {
            PrevSum_ = writeDelta(value.Sum, PrevSum_);
        }

        if (Mask_.HasMin()) {
            PrevMin_ = writeDelta(value.Min, PrevMin_);
        }

        if (Mask_.HasMax()) {
            PrevMax_ = writeDelta(value.Max, PrevMax_);
        }

        if (Mask_.HasLast()) {
            PrevLast_ = writeDelta(value.Last, PrevLast_);
        }
    }

    void WriteState(TBitWriter* writer) {
        writer->WriteInt8(Mask_, 5);
        Mask_ = {};

        // XXX: must be zig-zag encoded, but kept for backward compatibility with Java version
        writer->WriteVarInt64Mode(static_cast<ui64>(PrevCount_));
        PrevCount_ = 0;

        writer->WriteVarInt64Mode(static_cast<ui64>(PrevSum_));
        PrevSum_ = 0;

        writer->WriteVarInt64Mode(static_cast<ui64>(PrevMin_));
        PrevMin_ = 0;

        writer->WriteVarInt64Mode(static_cast<ui64>(PrevMax_));
        PrevMax_ = 0;

        writer->WriteVarInt64Mode(static_cast<ui64>(PrevLast_));
        PrevLast_ = 0;
    }

private:
    i64 PrevCount_{0};
    i64 PrevSum_{0};
    i64 PrevMin_{0};
    i64 PrevMax_{0};
    i64 PrevLast_{0};
    TSummaryMask Mask_;
};

/**
 * Integer summary timeseries decoder.
 */
class TSummaryIntTsDecoder: public TBaseTsDecoder<TSummaryIntTsDecoder, NValue::TSummaryInt> {
    using TBase = TBaseTsDecoder<TSummaryIntTsDecoder, NValue::TSummaryInt>;
    friend TBase;
    static constexpr auto EmptyValue = TSummaryIntPoint::EmptyValue;
public:
    TSummaryIntTsDecoder(TColumnSet columns, TBitSpan data)
        : TBase{columns, data}
    {
    }

    static TSummaryIntTsDecoder Simple(TBitSpan data) {
        return TSummaryIntTsDecoder{TSummaryIntPoint::SimpleColumns, data};
    }

    static TSummaryIntTsDecoder Aggr(TBitSpan data) {
        return TSummaryIntTsDecoder{TSummaryIntPoint::AggrColumns, data};
    }

private:
    void DecodeCommand(TBitReader* reader) {
        Mask_ = TSummaryMask{reader->ReadInt8(5)};
    }

    void DecodeValue(TBitReader* reader, NValue::TSummaryInt* value) {
        auto readDelta = [reader]() -> i64 {
            auto value = reader->ReadVarInt64Mode();
            Y_ENSURE(value.has_value(), "cannot read ISUMMARY field");
            return ZigZagDecode64(*value);
        };

        if (Mask_.HasCount()) {
            PrevCount_ += readDelta();
        }

        if (Mask_.HasSum()) {
            PrevSum_ += readDelta();
        }

        i64 min;
        if (Mask_.HasMin()) {
            PrevMin_ += readDelta();
            min = PrevMin_;
        } else {
            min = EmptyValue.Min;
        }

        i64 max;
        if (Mask_.HasMax()) {
            PrevMax_ += readDelta();
            max = PrevMax_;
        } else {
            max = EmptyValue.Max;
        }

        if (Mask_.HasLast()) {
            PrevLast_ += readDelta();
        }

        value->CountValue = PrevCount_;
        value->Sum = PrevSum_;
        value->Min = min;
        value->Max = max;
        value->Last = PrevLast_;
    }

    void Reset() {
        Mask_ = {};
        PrevCount_ = 0;
        PrevSum_ = 0;
        PrevMin_ = 0;
        PrevMax_ = 0;
        PrevLast_ = 0;
    }

private:
    i64 PrevCount_{0};
    i64 PrevSum_{0};
    i64 PrevMin_{0};
    i64 PrevMax_{0};
    i64 PrevLast_{0};
    TSummaryMask Mask_;
};

} // namespace NSolomon::NTs
