#include "time_codec.h"

#include <util/generic/yexception.h>

namespace NSolomon::NTs {

constexpr size_t TimeBitsPerMode[] = { 0, 4, 8, 12, 16, 24, 32, 64 };

void TTimeEncoder::Encode(TBitWriter* writer, TInstant time) {
    Y_VERIFY_DEBUG(time != TInstant::Zero(), "ts=0 is not supported");

    if (Y_UNLIKELY(!Prev_ || DeltaMillis_ == -1)) {
        // first two point encoded as is, because we can't calculate delta of delta right now
        writer->WriteInt64(time.MilliSeconds());

        if (Prev_) {
            DeltaMillis_ = static_cast<i64>(time.MilliSeconds()) - static_cast<i64>(Prev_.MilliSeconds());
        }

        Prev_ = time;
        return;
    }

    i64 deltaMillis = static_cast<i64>(time.MilliSeconds()) - static_cast<i64>(Prev_.MilliSeconds());
    if (deltaMillis == DeltaMillis_) {
        writer->WriteBit(false);
        Prev_ = time;
        return;
    }

    i64 deltaOfDelta = deltaMillis - DeltaMillis_;
    if (Y_UNLIKELY(!Millis_ && (deltaOfDelta % 1000 != 0))) {
        writer->WriteInt8(0xff);
        Millis_ = true;
        // could probably store more information here
    }

    i64 ddw = Y_UNLIKELY(Millis_) ? deltaOfDelta : deltaOfDelta / 1000;
    writer->WriteVarInt64Mode(ZigZagEncode64(ddw));

    DeltaMillis_ = deltaMillis;
    Prev_ = time;
}

TInstant TTimeDecoder::Decode(TBitReader* reader) {
    TInstant next;
    if (Y_UNLIKELY(!Prev_ || !PrevPrev_)) {
        Y_ENSURE(reader->Left() >= 64, "cannot read time, left(" << reader->Left() << ") < 64");
        next = TInstant::MilliSeconds(reader->ReadInt64());
    } else {
        Y_ENSURE(reader->Left() > 0, "cannot read time mode, left(" << reader->Left() << ')');
        ui8 mode = reader->ReadOnes(std::size(TimeBitsPerMode));
        if (Y_UNLIKELY(mode == std::size(TimeBitsPerMode))) {
            Y_ENSURE(!Millis_, "millis marker was already read");
            Millis_ = true;
            Y_ENSURE(reader->Left() > 0, "cannot read time mode, left(" << reader->Left() << ')');
            mode = reader->ReadOnes(8);
        }

        ui64 ddz = 0;
        if (size_t bits = TimeBitsPerMode[mode]; Y_UNLIKELY(bits != 0)) {
            Y_ENSURE(reader->Left() >= bits, "cannot read next time, left (" << reader->Left() << ") < bits(" << bits << ')');
            ddz = (bits == 64) ? reader->ReadInt64() : reader->ReadInt64(bits);
        }

        i64 dd = Y_UNLIKELY(Millis_) ? ZigZagDecode64(ddz) : ZigZagDecode64(ddz) * 1000;
        i64 result = Prev_.MilliSeconds() + dd + Prev_.MilliSeconds() - PrevPrev_.MilliSeconds();

        Y_ENSURE(result >= 0, "read negative time: " << result);
        next = TInstant::MilliSeconds(result);
    }

    PrevPrev_ = Prev_;
    return Prev_ = next;
}

} // namespace NSolomon::NTs
