#include <solomon/libs/cpp/ts_codec/bit_buffer.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NTs;

template <typename T>
class TBitStorageTest: public ::testing::Test {
public:
    using TStorage = T;
};

using TStorageTypes = ::testing::Types<TBitBuffer, TBitString>;
TYPED_TEST_SUITE(TBitStorageTest, TStorageTypes);

TYPED_TEST(TBitStorageTest, Empty) {
    using TStorage = typename TestFixture::TStorage;

    TStorage buf;
    ASSERT_EQ(buf.Size(), size_t(0));
    ASSERT_EQ(buf.Capacity(), size_t(0));
}

TYPED_TEST(TBitStorageTest, PushBack) {
    using TStorage = typename TestFixture::TStorage;

    TStorage buf;
    buf.PushBack(true);
    buf.PushBack(false);
    buf.PushBack(true);
    buf.PushBack(false);

    ASSERT_EQ(buf.Size(), size_t(4));
    ASSERT_GE(buf.Capacity(), size_t(8));
    EXPECT_TRUE(buf[0]);
    EXPECT_FALSE(buf[1]);
    EXPECT_TRUE(buf[2]);
    EXPECT_FALSE(buf[3]);

    buf.PushBack(false);
    buf.PushBack(true);
    buf.PushBack(true);
    buf.PushBack(false);

    ASSERT_EQ(buf.Size(), size_t(8));
    ASSERT_GE(buf.Capacity(), size_t(8));
    EXPECT_FALSE(buf[4]);
    EXPECT_TRUE(buf[5]);
    EXPECT_TRUE(buf[6]);
    EXPECT_FALSE(buf[7]);

    buf.PushBack(true);

    ASSERT_EQ(buf.Size(), size_t(9));
    ASSERT_GE(buf.Capacity(), size_t(16));
    EXPECT_TRUE(buf[8]);
}

TYPED_TEST(TBitStorageTest, FromString) {
    using TStorage = typename TestFixture::TStorage;

    auto empty = TStorage::FromString("");
    ASSERT_EQ(empty.Size(), size_t(0));
    ASSERT_EQ(empty.Capacity(), size_t(0));

    auto one = TStorage::FromString("1");
    ASSERT_EQ(one.Size(), size_t(1));
    ASSERT_GE(one.Capacity(), size_t(8));
    EXPECT_TRUE(one[0]);

    auto two = TStorage::FromString("1010 1001");
    ASSERT_EQ(two.Size(), size_t(8));
    ASSERT_GE(two.Capacity(), size_t(8));
    EXPECT_TRUE(two[0]);
    EXPECT_FALSE(two[1]);
    EXPECT_TRUE(two[2]);
    EXPECT_FALSE(two[3]);
    EXPECT_TRUE(two[4]);
    EXPECT_FALSE(two[5]);
    EXPECT_FALSE(two[6]);
    EXPECT_TRUE(two[7]);
}

TYPED_TEST(TBitStorageTest, ResizeUp) {
    using TStorage = typename TestFixture::TStorage;

    {
        auto small = TStorage::FromString("101");
        ASSERT_EQ(small.Size(), size_t(3));
        ASSERT_GE(small.Capacity(), size_t(8));
        EXPECT_TRUE(small[0]);
        EXPECT_FALSE(small[1]);
        EXPECT_TRUE(small[2]);

        small.Resize(5);

        ASSERT_EQ(small.Size(), size_t(5));
        ASSERT_GE(small.Capacity(), size_t(8));
        EXPECT_TRUE(small[0]);
        EXPECT_FALSE(small[1]);
        EXPECT_TRUE(small[2]);
        EXPECT_FALSE(small[3]);
        EXPECT_FALSE(small[4]);
    }

    {
        auto big = TStorage::FromString("10101010 10101010 10100101 01010101 01010101");
        ASSERT_EQ(big.Size(), size_t(40));
        ASSERT_GE(big.Capacity(), size_t(40));

        big.Resize(50);

        ASSERT_EQ(big.Size(), size_t(50));
        ASSERT_GE(big.Capacity(), size_t(56));
    }
}

TYPED_TEST(TBitStorageTest, ResizeDown) {
    using TStorage = typename TestFixture::TStorage;

    {
        auto small = TStorage::FromString("10101");
        ASSERT_EQ(small.Size(), size_t(5));
        EXPECT_TRUE(small[0]);
        EXPECT_FALSE(small[1]);
        EXPECT_TRUE(small[2]);
        EXPECT_FALSE(small[3]);
        EXPECT_TRUE(small[4]);

        small.Resize(3);

        ASSERT_EQ(small.Size(), size_t(3));
        ASSERT_GE(small.Capacity(), size_t(8));
        EXPECT_TRUE(small[0]);
        EXPECT_FALSE(small[1]);
        EXPECT_TRUE(small[2]);

        small.Resize(0);

        ASSERT_EQ(small.Size(), size_t(0));
        ASSERT_GE(small.Capacity(), size_t(8));
    }

    {
        auto big = TStorage::FromString("10101010 10101010 10100101 01010101 01010101");
        ASSERT_EQ(big.Size(), size_t(40));
        ASSERT_GE(big.Capacity(), size_t(40));

        big.Resize(30);

        ASSERT_EQ(big.Size(), size_t(30));
        ASSERT_GE(big.Capacity(), size_t(40));

        big.Resize(0);

        ASSERT_EQ(big.Size(), size_t(0));
        ASSERT_GE(big.Capacity(), size_t(40));
    }
}

TYPED_TEST(TBitStorageTest, Clear) {
    using TStorage = typename TestFixture::TStorage;

    auto buf = TStorage::FromString("00010101 01");
    ASSERT_EQ(buf.Size(), size_t(10));
    ASSERT_GE(buf.Capacity(), size_t(16));

    buf.Clear();

    ASSERT_EQ(buf.Size(), size_t(0));
    ASSERT_GE(buf.Capacity(), size_t(16));
}

TYPED_TEST(TBitStorageTest, ToSpan) {
    using TStorage = typename TestFixture::TStorage;

    auto buf = TStorage::FromString("1010 1110 1011 0010 10");
    auto span = static_cast<TBitSpan>(buf);

    ASSERT_EQ(buf.Data(), span.Data());
    ASSERT_EQ(buf.Size(), span.Size());
    ASSERT_EQ(buf.Size(), size_t(18));
}

TEST(TBitStorageTest, MovedStorage) {
    TBitBuffer buf{TBuffer("some data", 9)};
    ASSERT_EQ(buf.SizeBytes(), 9u);

    TBitString str{TString("some data")};
    ASSERT_EQ(buf.SizeBytes(), 9u);
}
