#include <solomon/libs/cpp/ts_codec/bit_reader.h>
#include <solomon/libs/cpp/ts_codec/ut/helpers.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <cmath>

using namespace NSolomon::NTs;

TEST(TBitReaderTest, ReadBit) {
    TBuffer buf = ParseBin("1011 0011");
    TBitReader r{buf.data(), 8};

    ASSERT_EQ(8u, r.Left());

    ASSERT_TRUE(r.ReadBit());  // 1
    ASSERT_FALSE(r.ReadBit()); // 0
    ASSERT_TRUE(r.ReadBit());  // 1
    ASSERT_TRUE(r.ReadBit());  // 1

    ASSERT_FALSE(r.ReadBit()); // 0
    ASSERT_FALSE(r.ReadBit()); // 0
    ASSERT_TRUE(r.ReadBit());  // 1
    ASSERT_TRUE(r.ReadBit());  // 1

    ASSERT_EQ(0u, r.Left());
}

TEST(TBitReaderTest, ReadInt8_Aligned) {
    TBuffer buf = ParseBin("10101000 01010001");
    TBitReader r{buf.data(), 16};

    ASSERT_EQ(16u, r.Left());

    ASSERT_EQ(0b00010101, r.ReadInt8());
    ASSERT_EQ(0b10001010, r.ReadInt8());

    ASSERT_EQ(0u, r.Left());
}

TEST(TBitReaderTest, ReadInt8_Unaligned) {
    TBuffer buf = ParseBin("1 10101000 01010001");
    TBitReader r{buf.data(), 17};

    ASSERT_EQ(17u, r.Left());

    ASSERT_TRUE(r.ReadBit());
    ASSERT_EQ(0b00010101, r.ReadInt8());
    ASSERT_EQ(0b10001010, r.ReadInt8());

    ASSERT_EQ(0u, r.Left());
}

TEST(TBitReaderTest, ReadInt32_Aligned) {
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 40};

    ASSERT_EQ(40u, r.Left());
    ASSERT_EQ(0b10100110'10110110'10000110'01010010, r.ReadInt32());
    ASSERT_EQ(8u, r.Left());
}

TEST(TBitReaderTest, ReadInt32_Unaligned) {
    TBuffer buf = ParseBin("1 01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 41};

    ASSERT_EQ(41u, r.Left());
    ASSERT_TRUE(r.ReadBit());
    ASSERT_EQ(0b10100110'10110110'10000110'01010010, r.ReadInt32());
    ASSERT_EQ(8u, r.Left());
}

TEST(TBitReaderTest, ReadInt64_Aligned) {
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100 01001010 01100001 01101101 01100101");
    TBitReader r{buf.data(), 72};

    ASSERT_EQ(72u, r.Left());
    ASSERT_EQ(
            0b10110110'10000110'01010010'00110110'10100110'10110110'10000110'01010010ull,
            r.ReadInt64());
    ASSERT_EQ(8u, r.Left());
}

TEST(TBitReaderTest, ReadInt64_Unaligned) {
    TBuffer buf = ParseBin("1 01001010 01100001 01101101 01100101 01101100 01001010 01100001 01101101 01100101");
    TBitReader r{buf.data(), 73};

    ASSERT_EQ(73u, r.Left());
    ASSERT_TRUE(r.ReadBit());
    ASSERT_EQ(
            0b10110110'10000110'01010010'00110110'10100110'10110110'10000110'01010010ull,
            r.ReadInt64());
    ASSERT_EQ(8u, r.Left());
}

TEST(TBitReaderTest, ReadDouble) {
    using Double = std::numeric_limits<double>;

    // bits string -> expected value
    std::map<TString, double> data = {
            {"1 00000000 00000000 00000000 00000000 00000000 00000000 00001000 00000000", Double::min()},
            {"1 11111111 11111111 11111111 11111111 11111111 11111111 11110111 11111110", Double::max()},
            {"1 00000000 00000000 00000000 00000000 00000000 00000000 00001111 11111110", Double::infinity()},
            {"1 00000000 00000000 00000000 00000000 00000000 00000000 00001111 11111111", -Double::infinity()},
            {"1 00000000 00000000 00000000 00000000 00000000 00000000 00011111 11111110", Double::quiet_NaN()},
            {"1 10010110 11101010 00101000 11010001 01010000 11111101 10100000 00000010", M_E},
            {"1 00011000 10110100 00100010 00101010 11011111 10000100 10010000 00000010", M_PI},
    };

    for (const auto& [text, value]: data) {
        TBuffer buf = ParseBin(text);
        TBitReader r{buf.data(), 65};

        ASSERT_EQ(65u, r.Left());

        ASSERT_TRUE(r.ReadBit());
        ASSERT_EQ(64u, r.Left());

        if (std::isnan(value)) {
            ASSERT_TRUE(std::isnan(r.ReadDouble()));
        } else {
            // exact match check
            ASSERT_EQ(value, r.ReadDouble()) << "text: " << text;
        }

        ASSERT_EQ(0u, r.Left());
    }
}

TEST(TBitReaderTest, ReadInt8Bits_Aligned) {
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 40};

    // number of bits to read -> expected value
    std::map<size_t, ui32> data = {
            {  0, 0b0 },
            {  1, 0b0 },
            {  2, 0b10 },
            {  3, 0b010 },
            {  4, 0b0010 },
            {  5, 0b10010 },
            {  6, 0b010010 },
            {  7, 0b1010010 },
            {  8, 0b01010010 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(0);

        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        ASSERT_EQ(value, r.ReadInt8(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos()) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt8Bits_Unaligned) {
    TBuffer buf = ParseBin("10 01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 42};

    // number of bits to read -> expected value
    std::map<size_t, ui32> data = {
            {  0, 0b0 },
            {  1, 0b0 },
            {  2, 0b10 },
            {  3, 0b010 },
            {  4, 0b0010 },
            {  5, 0b10010 },
            {  6, 0b010010 },
            {  7, 0b1010010 },
            {  8, 0b01010010 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(0);
        ASSERT_TRUE(r.ReadBit());
        ASSERT_FALSE(r.ReadBit());

        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        size_t pos = r.Pos();
        ASSERT_EQ(value, r.ReadInt8(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos() - pos) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt8Bits_AtEnd) {
    //                                                            / read position
    //                                                           V
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 40};

    // number of bits to read -> expected value
    std::map<size_t, ui32> data = {
            {  0, 0b0 },
            {  1, 0b1 },
            {  2, 0b11 },
            {  3, 0b011 },
            {  4, 0b1011 },
            {  5, 0b11011 },
            {  6, 0b011011 },
            {  7, 0b0011011 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(33);
        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        size_t pos = r.Pos();
        ASSERT_EQ(value, r.ReadInt8(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos() - pos) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt32Bits_Aligned) {
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 40};

    // number of bits to read -> expected value
    std::map<size_t, ui32> data = {
            {  0, 0b0 },
            {  1, 0b0 },
            {  2, 0b10 },
            {  3, 0b010 },
            {  4, 0b0010 },
            {  8, 0b01010010 },
            { 16, 0b10000110'01010010 },
            { 24, 0b10110110'10000110'01010010 },
            { 31, 0b00100110'10110110'10000110'01010010 },
            { 32, 0b10100110'10110110'10000110'01010010 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(0);

        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        ASSERT_EQ(value, r.ReadInt32(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos()) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt32Bits_Unaligned) {
    TBuffer buf = ParseBin("10 01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 42};

    // number of bits to read -> expected value
    std::map<size_t, ui32> data = {
            {  0, 0b0 },
            {  1, 0b0 },
            {  2, 0b10 },
            {  3, 0b010 },
            {  4, 0b0010 },
            {  8, 0b01010010 },
            { 16, 0b10000110'01010010 },
            { 24, 0b10110110'10000110'01010010 },
            { 31, 0b00100110'10110110'10000110'01010010 },
            { 32, 0b10100110'10110110'10000110'01010010 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(0);
        ASSERT_TRUE(r.ReadBit());
        ASSERT_FALSE(r.ReadBit());

        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        size_t pos = r.Pos();
        ASSERT_EQ(value, r.ReadInt32(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos() - pos) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt32Bits_AtEnd) {
    //                                             / read position
    //                                            V
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100");
    TBitReader r{buf.data(), 40};

    // number of bits to read -> expected value
    std::map<size_t, ui32> data = {
            {  0, 0b0 },
            {  1, 0b1 },
            {  2, 0b11 },
            {  3, 0b011 },
            {  4, 0b1011 },
            {  8, 0b01101011 },
            { 16, 0b01101010'01101011 },
            { 20, 0b0011'01101010'01101011 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(20);
        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        size_t pos = r.Pos();
        ASSERT_EQ(value, r.ReadInt32(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos() - pos) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt64Bits_Aligned) {
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100 01001010 01100001 01101101 01100101");
    TBitReader r{buf.data(), 72};

    // number of bits to read -> expected value
    std::map<size_t, ui64> data = {
            {  0, 0b0 },
            {  1, 0b0 },
            {  2, 0b10 },
            {  3, 0b010 },
            {  4, 0b0010 },
            {  8, 0b01010010 },
            { 16, 0b10000110'01010010 },
            { 24, 0b10110110'10000110'01010010 },
            { 32, 0b10100110'10110110'10000110'01010010 },
            { 40, 0b00110110'10100110'10110110'10000110'01010010 },
            { 48, 0b01010010'00110110'10100110'10110110'10000110'01010010 },
            { 56, 0b10000110'01010010'00110110'10100110'10110110'10000110'01010010 },
            { 63, 0b00110110'10000110'01010010'00110110'10100110'10110110'10000110'01010010 },
            { 64, 0b10110110'10000110'01010010'00110110'10100110'10110110'10000110'01010010 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(0);

        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        ASSERT_EQ(value, r.ReadInt64(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos()) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt64Bits_Unligned) {
    TBuffer buf = ParseBin("10 01001010 01100001 01101101 01100101 01101100 01001010 01100001 01101101 01100101");
    TBitReader r{buf.data(), 72};

    // number of bits to read -> expected value
    std::map<size_t, ui64> data = {
            {  0, 0b0 },
            {  1, 0b0 },
            {  2, 0b10 },
            {  3, 0b010 },
            {  4, 0b0010 },
            {  8, 0b01010010 },
            { 16, 0b10000110'01010010 },
            { 24, 0b10110110'10000110'01010010 },
            { 32, 0b10100110'10110110'10000110'01010010 },
            { 40, 0b00110110'10100110'10110110'10000110'01010010 },
            { 48, 0b01010010'00110110'10100110'10110110'10000110'01010010 },
            { 56, 0b10000110'01010010'00110110'10100110'10110110'10000110'01010010 },
            { 63, 0b00110110'10000110'01010010'00110110'10100110'10110110'10000110'01010010 },
            { 64, 0b10110110'10000110'01010010'00110110'10100110'10110110'10000110'01010010 },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(0);
        ASSERT_TRUE(r.ReadBit());
        ASSERT_FALSE(r.ReadBit());

        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        size_t pos = r.Pos();
        ASSERT_EQ(value, r.ReadInt64(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos() - pos) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadInt64Bits_AtEnd) {
    //                                                              / read position
    //                                                             V
    TBuffer buf = ParseBin("01001010 01100001 01101101 01100101 01101100 01001010 01100001 01101101 01100101");
    TBitReader r{buf.data(), 72};

    // number of bits to read -> expected value
    std::map<size_t, ui64> data = {
            {  0, 0b0 },
            {  1, 0b0 },
            {  2, 0b10 },
            {  3, 0b110 },
            {  4, 0b0110 },
            {  8, 0b01000110 },
            { 16, 0b11001010'01000110 },
            { 24, 0b11010000'11001010'01000110 },
            { 32, 0b11010110'11010000'11001010'01000110ul },
            { 37, 0b00010100'11010110'11010000'11001010'01000110ul },
    };

    for (const auto [bits, value]: data) {
        r.SetPos(35);
        ASSERT_GE(r.Left(), bits) << "bits: " << bits;

        size_t pos = r.Pos();
        ASSERT_EQ(value, r.ReadInt64(bits)) << "bits: " << bits;
        ASSERT_EQ(bits, r.Pos() - pos) << "bits: " << bits;
    }
}

TEST(TBitReaderTest, ReadVarInt32) {
    TBuffer buf = ParseBin("01 00000000 11000000 11011111 11111111 11111111 11111111 11110000");
    TBitReader r{buf.data(), 58};

    ASSERT_FALSE(r.ReadBit());
    ASSERT_TRUE(r.ReadBit());

    auto v1 = r.ReadVarInt32();
    ASSERT_TRUE(v1.has_value());
    ASSERT_EQ(0u, *v1);

    auto v2 = r.ReadVarInt32();
    ASSERT_TRUE(v2.has_value());
    ASSERT_EQ(3u, *v2);

    auto v3 = r.ReadVarInt32();
    ASSERT_TRUE(v3.has_value());
    ASSERT_EQ(ui32(-5), *v3);

    auto v4 = r.ReadVarInt32();
    ASSERT_FALSE(v4.has_value());
}

TEST(TBitReaderTest, ReadVarInt64) {
    TBuffer buf = ParseBin("01 00000000 11000000 "
                           "11011111 11111111 11111111 11111111 11111111"
                           "11111111 11111111 11111111 11111111 1000000");
    TBitReader r{buf.data(), 98};

    ASSERT_FALSE(r.ReadBit());
    ASSERT_TRUE(r.ReadBit());

    auto v1 = r.ReadVarInt64();
    ASSERT_TRUE(v1.has_value());
    ASSERT_EQ(0u, *v1);

    auto v2 = r.ReadVarInt64();
    ASSERT_TRUE(v2.has_value());
    ASSERT_EQ(3u, *v2);

    auto v3 = r.ReadVarInt64();
    ASSERT_TRUE(v3.has_value());
    ASSERT_EQ(ui64(-5), *v3);

    auto v4 = r.ReadVarInt64();
    ASSERT_FALSE(v4.has_value());
}

TEST(TBitReaderTest, ReadOnes) {
    TBuffer buf = ParseBin("0 10 110 111 0");
    TBitReader r{buf.data(), 10};

    ASSERT_EQ(0u, r.ReadOnes(3));
    ASSERT_EQ(1u, r.ReadOnes(3));
    ASSERT_EQ(2u, r.ReadOnes(3));
    ASSERT_EQ(3u, r.ReadOnes(3));

    ASSERT_EQ(0u, r.ReadOnes(4));

    ASSERT_EQ(0u, r.Left());
}

TEST(TBitReaderTest, WriteVarInt32Mode) {
    std::vector<std::pair<ui32, TStringBuf>> data = {
            {          0, "0"                                    },
            {         10, "100101"                               },
            {        100, "11000100110"                          },
            {      1'000, "1110000101111100"                     },
            {     10'000, "1111000001000111001000000000000000000"},
            {    100'000, "1111000000101011000011000000000000000"},
            {100'000'000, "1111000000000100001111010111110100000"},
    };

    for (auto [expected, bits]: data) {
        auto [buf, len] = ParseBin2(bits);
        TBitReader r{buf.data(), len};
        auto value = r.ReadVarInt32Mode();
        ASSERT_TRUE(value.has_value());
        ASSERT_EQ(*value, expected);
    }
}

TEST(TBitReaderTest, WriteVarInt64Mode) {
    std::vector<std::pair<ui64, TStringBuf>> data = {
            {          0, "0"                                      },
            {         10, "100101"                                 },
            {        100, "11000100110"                            },
            {      1'000, "1110000101111100"                       },
            {     10'000, "111100000100011100100"                  },
            {    100'000, "111110000001010110000110000000"         },
            {100'000'000, "111111000000000100001111010111110100000"},
    };

    for (auto [expected, bits]: data) {
        auto [buf, len] = ParseBin2(bits);
        TBitReader r{buf.data(), len};
        auto value = r.ReadVarInt64Mode();
        ASSERT_TRUE(value.has_value()) << "bits: " << bits;
        ASSERT_EQ(*value, expected) << "bits: " << bits;
    }
}
