#include <solomon/libs/cpp/ts_codec/bit_writer.h>
#include <solomon/libs/cpp/ts_codec/ut/helpers.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>

#include <cmath>

using namespace NSolomon::NTs;

TEST(TBitWriterTest, WriteBit) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    // first tetrade (0101)
    w.WriteBit(true);
    ASSERT_EQ("01", ToHex(buf));
    w.WriteBit(false);
    ASSERT_EQ("01", ToHex(buf));
    w.WriteBit(true);
    ASSERT_EQ("05", ToHex(buf));
    w.WriteBit(false);
    ASSERT_EQ("05", ToHex(buf));

    // second tetrade (0101)
    w.WriteBit(true);
    ASSERT_EQ("15", ToHex(buf));
    w.WriteBit(false);
    ASSERT_EQ("15", ToHex(buf));
    w.WriteBit(true);
    ASSERT_EQ("55", ToHex(buf));
    w.WriteBit(false);
    ASSERT_EQ("55", ToHex(buf));

    // third tetrade (1010)
    w.WriteBit(false);
    ASSERT_EQ("5500", ToHex(buf));
    w.WriteBit(true);
    ASSERT_EQ("5502", ToHex(buf));
    w.WriteBit(false);
    ASSERT_EQ("5502", ToHex(buf));
    w.WriteBit(true);
    ASSERT_EQ("550A", ToHex(buf));
}

TEST(TBitWriterTest, WriteBitStress) {
    SetRandomSeed(17);

    for (size_t i = 0; i < 10'000; i++) {
        TBitBuffer expected;
        FillRandomly(40, &expected);

        TBitBuffer buf;
        FillRandomly(40, &buf); // garbage must be properly rewritten

        TBitWriter w{&buf};
        for (size_t j = 0; j < expected.Size(); j++) {
            w.WriteBit(expected[j]);
        }

        w.Flush();
        ASSERT_EQ(ToBin(expected), ToBin(buf)) << "i=" << i;
    }
}

TEST(TBitWriterTest, WriteInt8) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteInt8(0b11111111);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11111111"));

    w.WriteInt8(0b11110000);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11111111 00001111"));

    w.WriteInt8(0b00001111);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11111111 00001111 11110000"));

    w.AlignToByte();
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11111111 00001111 11110000 000000"));

    w.WriteInt8(0b10101010);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11111111 00001111 11110000 000000 01010101"));
}

TEST(TBitWriterTest, WriteInt32) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteInt32(0b01000101'00100100'01011010'10101011);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010101 01011010 00100100 10100010"));
}

TEST(TBitWriterTest, WriteInt64) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteInt64(0b01010101'00100100'01011010'10101011'00000101'00100100'01011010'10100011ULL);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11000101 01011010 00100100 10100000 11010101 01011010 00100100 10101010"));
}

TEST(TBitWriterTest, WriteDouble) {
    using Double = std::numeric_limits<double>;

    std::map<TString, double> data = {
            {"01 00000000 00000000 00000000 00000000 00000000 00000000 00001000 00000000", Double::min()},
            {"01 11111111 11111111 11111111 11111111 11111111 11111111 11110111 11111110", Double::max()},
            {"01 00000000 00000000 00000000 00000000 00000000 00000000 00001111 11111110", Double::infinity()},
            {"01 00000000 00000000 00000000 00000000 00000000 00000000 00001111 11111111", -Double::infinity()},
            {"01 00000000 00000000 00000000 00000000 00000000 00000000 00011111 11111110", Double::quiet_NaN()},
            {"01 10010110 11101010 00101000 11010001 01010000 11111101 10100000 00000010", M_E},
            {"01 00011000 10110100 00100010 00101010 11011111 10000100 10010000 00000010", M_PI},
    };

    for (const auto& [text, value]: data) {
        TBitBuffer buf;
        TBitWriter w{&buf};

        w.WriteBit(false);
        w.WriteBit(true);
        w.Flush();
        ASSERT_TRUE(IsSameBuf(buf, "01"));

        w.WriteDouble(value);
        w.Flush();
        ASSERT_TRUE(IsSameBuf(buf, text));
    }
}

TEST(TBitWriterTest, WriteInt8Bits) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteInt8(0b01010101, 1);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 1"));

    w.WriteInt8(0b01010101, 2);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 110"));

    w.WriteInt8(0b01010101, 3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 110101"));

    w.WriteInt8(0b01010101, 4);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10"));

    w.WriteInt8(0b01010101, 5);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 1010101"));

    w.WriteInt8(0b01010101, 6);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010"));

    w.WriteInt8(0b01010101, 7);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 0101"));

    w.AlignToByte();
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 010100"));

    w.WriteInt8(0b01010101, 5);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 010100 10101"));
}

TEST(TBitWriterTest, WriteInt32Bits) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteInt32(0x55555555, 1);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 1"));

    w.WriteInt32(0x55555555, 2);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 110"));

    w.WriteInt32(0x55555555, 3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 110101"));

    w.WriteInt32(0x55555555, 4);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10"));

    w.WriteInt32(0x55555555, 5);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 1010101"));

    w.WriteInt32(0x55555555, 6);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010"));

    w.WriteInt32(0x55555555, 7);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 0101"));

    w.WriteInt32(0x55555555, 8);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 01011010 1010"));

    w.AlignToByte();
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 01011010 101000"));

    w.WriteInt32(0x55555555, 31);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 01011010 101000"
                               "10101010 10101010 10101010 1010101"));
}

TEST(TBitWriterTest, WriteInt64Bits) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteInt64(0x55555555'55555555ull, 1);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 1"));

    w.WriteInt64(0x55555555'55555555ull, 2);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 110"));

    w.WriteInt64(0x55555555'55555555ull, 3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 110101"));

    w.WriteInt64(0x55555555'55555555ull, 4);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10"));

    w.WriteInt64(0x55555555'55555555ull, 5);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 1010101"));

    w.WriteInt64(0x55555555'55555555ull, 6);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010"));

    w.WriteInt64(0x55555555, 7);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 0101"));

    w.WriteInt64(0x55555555'55555555ull, 8);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 01011010 1010"));

    w.AlignToByte();
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 01011010 101000"));

    w.WriteInt64(0x55555555'55555555ull, 63);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 11010110 10101011 01010101 01011010 101000"
                               "10101010 10101010 10101010 10101010 10101010 10101010 10101010 1010101"));

}

TEST(TBitWriterTest, WriteVarInt32) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteVarInt32(0);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 00000000"));

    w.WriteVarInt32(3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 00000000 11000000"));

    w.WriteVarInt32(-5);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 00000000 11000000 11011111 11111111 11111111 11111111 11110000"));
}

TEST(TBitWriterTest, WriteVarInt64) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteVarInt64(0);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 00000000"));

    w.WriteVarInt64(3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 00000000 11000000"));

    w.WriteVarInt64(-5);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 00000000 11000000 "
                               "11011111 11111111 11111111 11111111 11111111"
                               "11111111 11111111 11111111 11111111 1000000"));
}

TEST(TBitWriterTest, WriteOnes) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    w.WriteBit(false);
    w.WriteBit(true);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01"));

    w.WriteOnes(0, 3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 0"));

    w.WriteOnes(1, 3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 0 10"));

    w.WriteOnes(2, 3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 0 10 110"));

    w.WriteOnes(3, 3);
    w.Flush();
    ASSERT_TRUE(IsSameBuf(buf, "01 0 10 110 111"));
}

TEST(TBitWriterTest, WriteVarInt32Mode) {
    std::vector<std::pair<ui32, TStringBuf>> data = {
            {          0, "0"                                    },
            {         10, "100101"                               },
            {        100, "11000100110"                          },
            {      1'000, "1110000101111100"                     },
            {     10'000, "1111000001000111001000000000000000000"},
            {    100'000, "1111000000101011000011000000000000000"},
            {100'000'000, "1111000000000100001111010111110100000"},
    };

    for (auto [value, bits]: data) {
        TBitBuffer buf;
        TBitWriter w{&buf};
        w.WriteVarInt32Mode(value);
        w.Flush();
        EXPECT_TRUE(IsSameBuf(buf, bits)) << "value=" << value;
    }
}

TEST(TBitWriterTest, WriteVarInt64Mode) {
    std::vector<std::pair<ui64, TStringBuf>> data = {
            {          0, "0"                                      },
            {         10, "100101"                                 },
            {        100, "11000100110"                            },
            {      1'000, "1110000101111100"                       },
            {     10'000, "111100000100011100100"                  },
            {    100'000, "111110000001010110000110000000"         },
            {100'000'000, "111111000000000100001111010111110100000"},
    };

    for (auto [value, bits]: data) {
        TBitBuffer buf;
        TBitWriter w{&buf};
        w.WriteVarInt64Mode(value);
        w.Flush();
        EXPECT_TRUE(IsSameBuf(buf, bits)) << "value=" << value;
    }
}
