#include <solomon/libs/cpp/ts_codec/count_codec.h>
#include <solomon/libs/cpp/ts_codec/ut/helpers.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>

using namespace NSolomon::NTs;

namespace {

std::vector<std::tuple<ui64, i64, TStringBuf>> Data = {
        // count, prev, encoded block
        {      0,   -1, "00000000"},
        {      1,    0, "100100"},
        {     10,    1, "11001001 000"},
        {    100,   10, "11000101 101"},
        {   1000,  100, "11100001 00001110"},
};

} // namespace

TEST(TCountCodecTest, InitState) {
    TCountEncoder encoder;
    EXPECT_EQ(encoder.Prev(), -1);
}

TEST(TCountCodecTest, Encode) {
    TCountEncoder encoder;

    for (const auto& [count, prev, expectedBits]: Data) {
        TBitBuffer buffer;
        TBitWriter w{&buffer};

        EXPECT_EQ(encoder.Prev(), prev) << "count=" << count;
        encoder.Encode(&w, count);

        EXPECT_EQ(encoder.Prev(), static_cast<i64>(count)) << "count=" << count;

        w.Flush();
        ASSERT_TRUE(IsSameBuf(buffer, expectedBits)) << "count=" << count;
    }
}

TEST(TCountCodecTest, Decode) {
    TCountDecoder decoder;

    for (const auto& [expectedCount, prev, bits]: Data) {
        auto [buf, len] = ParseBin2(bits);
        TBitReader r{buf.data(), len};

        ui64 count;
        ASSERT_NO_THROW({ count = decoder.Decode(&r); }) << "expectedCount=" << expectedCount;
        ASSERT_EQ(count, expectedCount);
    }
}

TEST(TCountCodecTest, EncodeDecodeStress) {
    SetRandomSeed(1137);

    TBitBuffer buf;
    TBitWriter w{&buf};
    TCountEncoder encoder;

    std::vector<ui64> expected;
    expected.reserve(1000);

    for (size_t i = 0; i < 1000; ++i) {
        ui64 count = RandomNumber(10'000'000u);
        expected.push_back(count);
        encoder.Encode(&w, count);
    }
    w.Flush();

    TCountDecoder decoder;
    TBitReader r{buf};

    for (size_t i = 0; r.Left() != 0; i++) {
        ui64 count;
        ASSERT_NO_THROW({ count = decoder.Decode(&r); }) << "expected=" << expected[i];
        ASSERT_EQ(count, expected[i]);
    }
}
