#include <solomon/libs/cpp/ts_codec/counter_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/ut/helpers.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>

using namespace NSolomon::NTs;

class TCounterTsCodecTest: public ::testing::Test {
public:
    void SetUp() override {
        Buf_ = {};
    }

    void Encode(std::vector<TLongPoint> points, TColumnSet columns = TLongPoint::AggrColumns) {
        Points_ = std::move(points);

        TBitWriter w{&Buf_};
        auto encoder = TCounterTsEncoder{columns, &w};
        for (const auto& p: Points_) {
            encoder.EncodePoint(p);
        }

        encoder.Flush();
    }

    void EncodeRandom(const std::function<i64(size_t)>& gen) {
        TBitWriter w{&Buf_};
        auto encoder = TCounterTsEncoder::Aggr(&w);

        TLongPoint p;
        p.Step = TDuration::Seconds(15);

        TInstant time = TInstant::ParseIso8601("2020-05-30T11:12:00Z");
        for (size_t i = 0; i < 500; ++i) {
            p.Time = time;
            p.Value = gen(i);
            encoder.EncodePoint(p);
            Points_.push_back(p);

            time += p.Step;
        }

        encoder.Flush();
    }

    void DecodeAndCheck(TColumnSet columns = TLongPoint::AggrColumns, TLongPoint point = {}) {
        auto decoder = TCounterTsDecoder{columns, Buf_};
        ASSERT_TRUE(decoder.HasNext());

        for (size_t i = 0; i < Points_.size(); ++i) {
            const auto& expected = Points_[i];

            bool hasNext;
            ASSERT_NO_THROW({ hasNext = decoder.NextPoint(&point); }) << "point #" << i;
            ASSERT_TRUE(hasNext) << "point #" << i;

            EXPECT_EQ(expected.Time, point.Time) << "point #" << i;
            EXPECT_EQ(expected.Step, point.Step) << "point #" << i;
            EXPECT_EQ(expected.Value, point.Value) << "point #" << i;

            EXPECT_EQ(expected.Merge, point.Merge) << "point #" << i;
            EXPECT_EQ(expected.Count, point.Count) << "point #" << i;
        }

        ASSERT_FALSE(decoder.HasNext());
    }

    static TLongPoint MakePoint(TStringBuf time, i64 value) {
        TLongPoint point;
        point.Time = TInstant::ParseIso8601(time);
        point.Value = value;
        return point;
    }

private:
    std::vector<TLongPoint> Points_;
    TBitBuffer Buf_;
};

TEST_F(TCounterTsCodecTest, SequentialGrow) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1),
            MakePoint("2020-05-30T11:12:15Z", 2),
            MakePoint("2020-05-30T11:12:30Z", 3),
            MakePoint("2020-05-30T11:12:45Z", 4),
            MakePoint("2020-05-30T11:13:00Z", 5),
    });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, UpAndDown) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1),
            MakePoint("2020-05-30T11:12:15Z", 8),
            MakePoint("2020-05-30T11:12:30Z", 2),
            MakePoint("2020-05-30T11:12:45Z", 7),
            MakePoint("2020-05-30T11:13:00Z", 3),
    });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, SameValue) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 42),
            MakePoint("2020-05-30T11:12:15Z", 42),
            MakePoint("2020-05-30T11:12:30Z", 42),
            MakePoint("2020-05-30T11:12:45Z", 42),
            MakePoint("2020-05-30T11:13:00Z", 42),
    });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, MinMax) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 0),
            MakePoint("2020-05-30T11:12:15Z", std::numeric_limits<i64>::min()), // 0 -> min
            MakePoint("2020-05-30T11:12:30Z", 0),
            MakePoint("2020-05-30T11:12:45Z", std::numeric_limits<i64>::max()), // 0 -> max
            MakePoint("2020-05-30T11:13:00Z", 0),
            MakePoint("2020-05-30T11:13:15Z", std::numeric_limits<i64>::min()), // min -> max
            MakePoint("2020-05-30T11:13:30Z", std::numeric_limits<i64>::max()),
            MakePoint("2020-05-30T11:13:45Z", 0),
            MakePoint("2020-05-30T11:14:00Z", std::numeric_limits<i64>::max()), // max -> min
            MakePoint("2020-05-30T11:14:15Z", std::numeric_limits<i64>::min()),
            MakePoint("2020-05-30T11:14:30Z", 0),
    });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, ConstZero) {
    EncodeRandom([](size_t) { return 0; });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, ConstOne) {
    EncodeRandom([](size_t) { return 1; });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, OneOrZero) {
    EncodeRandom([](size_t) { return RandomNumber<bool>() ? 0 : 1; });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, From_0_To_1000000) {
    EncodeRandom([](size_t) { return RandomNumber<ui64>(1000000); });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, AnyInt) {
    EncodeRandom([](size_t) { return RandomNumber<ui32>(); });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, AnyLong) {
    EncodeRandom([](size_t) { return RandomNumber<ui64>(); });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, Grow) {
    EncodeRandom([](size_t i) { return i; });
    DecodeAndCheck();
}

TEST_F(TCounterTsCodecTest, DirtyPoint) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 42),
            MakePoint("2020-05-30T11:12:15Z", 42),
            MakePoint("2020-05-30T11:12:30Z", 42),
            MakePoint("2020-05-30T11:12:45Z", 42),
            MakePoint("2020-05-30T11:13:00Z", 42),
    }, TLongPoint::SimpleColumns);

    TLongPoint point;

    // All of these should be overwritten when decoding
    point.Time = TInstant::Minutes(10);
    point.Step = TDuration::Minutes(30);
    point.Count = 10;
    point.Merge = true;

    DecodeAndCheck(TLongPoint::SimpleColumns, point);
}
