#include <solomon/libs/cpp/ts_codec/denom_codec.h>
#include <solomon/libs/cpp/ts_codec/ut/helpers.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>

using namespace NSolomon::NTs;

namespace {

std::vector<std::pair<ui64, TStringBuf>> Data = {
        // denom, next encoded block
        {      0, "000"},
        {      1, "010 10000000"},
        {     10, "010 01010000"},
        {    100, "010 00100110"},
        {    200, "010 00010011 10000000"},
        {    400, "010 00001001 11000000"},
        {  1'000, "100 10000000"},
        {  2'000, "100 01000000"},
        { 15'000, "100 11110000"},
        { 30'000, "100 01111000"},
        {300'000, "100 00110101 01000000"},
};

ui64 RandomDenom() {
    switch (RandomNumber(3u)) {
        case 0u: return 0u;
        case 1u: return 1u + RandomNumber(1000u);
        default: return 1000u * RandomNumber(1000u);
    }
};

} // namespace

TEST(TDenomCodecTest, Encode) {
    for (auto [value, bits]: Data) {
        TBitBuffer buf;
        TBitWriter w{&buf};
        DenomEncode(&w, value);
        w.Flush();
        EXPECT_TRUE(IsSameBuf(buf, bits)) << "value=" << value;
    }
}

TEST(TDenomCodecTest, Decode) {
    for (auto [expected, bits]: Data) {
        auto [buf, len] = ParseBin2(bits);
        TBitReader r{buf.data(), len};

        ui64 denom;
        ASSERT_NO_THROW({ denom = DenomDecode(&r); }) << "value=" << expected;
        EXPECT_EQ(expected, denom);
    }
}

TEST(TDenomCodecTest, EncodeDecodeStress) {
    SetRandomSeed(123);

    TBitBuffer buf;
    TBitWriter w{&buf};

    std::vector<ui64> expected;
    for (int i = 0; i < 1000; ++i) {
        ui64 denom = RandomDenom();
        expected.push_back(denom);
        DenomEncode(&w, denom);
    }

    w.Flush();

    TBitReader r{buf};
    for (size_t i = 0; r.Left() != 0; ++i) {
        ui64 denom = DenomDecode(&r);
        ASSERT_EQ(denom, expected[i]);
    }
}
